(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Sat Oct  3 17:47:29 PDT 1992 by meehan                   *)
(*      modified on Sun Jul 12 22:42:04 1992 by mhb                          *)
(*      modified on Tue Jun 16 22:56:07 PDT 1992 by muller                   *)

MODULE FVRuntime EXPORTS FVRuntime, FVTypes, FormsVBT;

(* This module contains the runtime code for FormsVBTs. *)

IMPORT AnchorSplit, AnyEvent, Axis, BooleanVBT, BorderedVBT, ButtonVBT,
       ChoiceVBT, ClipboardVBT, ColorName, FileBrowserVBT, Filename,
       FileStream, Filter, FlexShape, FlexVBT, Fmt, Font, GuardedBtnVBT,
       HighlightVBT, HVSplit, List, ListVBT, MenuSwitchVBT, Multi,
       NumericScrollerVBT, NumericVBT, PaintOp, PaintOpCache, Radio, Rd,
       RdUtils, ReactivityVBT, RGB, Rsrc, RTutils, Shadow,
       ShadowedFeedbackVBT, SmallIO, SourceVBT, Split, SplitterVBT, SwitchVBT,
       Sx, SxSymbol, Text, TextEditVBT, TextPort, TextRd, TextureVBT, TextVBT,
       Thread, TrillSwitchVBT, TSplit, TxtIntTbl, TxtRefTbl, TypescriptVBT,
       VBT, VBTClass, Wr, ZChassisVBT, ZChildVBT, ZSplit, ZSplitUtils;

FROM SmallIO IMPORT stderr;

<* PRAGMA LL *>

REVEAL
  T = Private BRANDED OBJECT
        mu: MUTEX;
        <* LL = mu *>
        getVBT     : TxtRefTbl.T;
        eventCount : CARDINAL              := 0;
        keyRec     : REF VBT.KeyRec;
        mouseRec   : REF VBT.MouseRec;
        positionRec: REF VBT.PositionRec;
        miscRec    : REF VBT.MiscRec;
        eventCode  : CARDINAL              := 0; (* typecode of event *)
        timeStamp  : VBT.TimeStamp;
        gensym                             := 0;
        raw                                := FALSE;
      OVERRIDES
        init         := InitFromText;
        initFromFile := InitFromFile;
        initFromSx   := InitFromSx;
        initFromRd   := InitFromRd;
        initFromRsrc := InitFromRsrc;
        snapshot     := Snapshot;
        restore      := Restore;
      END;

VAR cleanState: State;            (* CONST *)


(*************************** Creation *******************************)
  
PROCEDURE NewFromFile (filename: TEXT; raw := FALSE; path: Rsrc.Path := NIL): T
  RAISES {Error, Rd.Failure, Thread.Alerted} =
  BEGIN
    RETURN NEW (T).initFromFile (filename, raw, path)
  END NewFromFile;

PROCEDURE InitFromFile (fv      : T;
                        filename: TEXT;
                        raw                 := FALSE;
                        path    : Rsrc.Path := NIL    ): T
  RAISES {Error, Rd.Failure, Thread.Alerted} =
  VAR rd: Rd.T;
  BEGIN
    TRY
      rd := FileStream.OpenRead (Filename.ExpandTilde (filename));
      TRY RETURN InitFromRd (fv, rd, raw, path) FINALLY Rd.Close (rd) END
    EXCEPT
    | Filename.Error => RAISE Error ("Bad filename: " & filename)
    END
  END InitFromFile;

PROCEDURE InitFromText (fv         : T;
                        description: TEXT;
                        raw                    := FALSE;
                        path       : Rsrc.Path := NIL    ): T RAISES {Error} =
  <* FATAL Rd.Failure, Thread.Alerted *>
  BEGIN
    RETURN InitFromRd (fv, TextRd.New (description), raw, path)
  END InitFromText;

TYPE
  ReaderClosure = Thread.SizedClosure OBJECT
                    rd     : Rd.T;
                    errType: ErrType;
                    errArg : REFANY
                  OVERRIDES
                    apply := Read
                  END;
  ErrType = {ReadError, EndOfFile, Failure, Alerted};
 
PROCEDURE Read (rc: ReaderClosure): REFANY =
  VAR
    exp  : REFANY;
    gotIt         := FALSE;
  BEGIN
    TRY
      exp := Sx.Read (rc.rd, syntax := FVSyntax);
      gotIt := TRUE;
      IF Rd.EOF (rc.rd) THEN RETURN exp END;
      (* Check for extra garbage: *)
      EVAL Sx.Read (rc.rd, syntax := FVSyntax);
      RAISE Sx.ReadError ("extra characters on input")
    EXCEPT
    | Sx.ReadError (txt) => rc.errArg := txt; rc.errType := ErrType.ReadError
    | Rd.EndOfFile =>
        IF gotIt THEN RETURN exp END;
        rc.errType := ErrType.EndOfFile
    | Rd.Failure (ref) => rc.errArg := ref; rc.errType := ErrType.Failure
    | Thread.Alerted => rc.errType := ErrType.Alerted
    END;
    (* If there's an error, we return the ReaderClosure itself. *)
    RETURN rc
  END Read;
               
PROCEDURE InitFromRd (fv: T; rd: Rd.T; raw := FALSE; path: Rsrc.Path := NIL): T
  RAISES {Error, Rd.Failure, Thread.Alerted} =
  BEGIN
    TYPECASE
        Thread.Join (
          Thread.Fork (NEW (ReaderClosure, rd := rd, stackSize := 10000))) OF
    | ReaderClosure (rc) =>
        CASE rc.errType OF
        | ErrType.ReadError => RAISE Error ("Sx.ReadError: " & rc.errArg)
        | ErrType.EndOfFile => RAISE Error ("End of input")
        | ErrType.Failure => RAISE Rd.Failure (rc.errArg)
        | ErrType.Alerted => RAISE Thread.Alerted
        END
    | REFANY (desc) => RETURN InitFromSx (fv, desc, raw, path)
    END
  END InitFromRd;

PROCEDURE InitFromRsrc (fv: T; name: TEXT; path: Rsrc.Path; raw := FALSE): T
  RAISES {Error, Rd.Failure, Rsrc.NotFound, Thread.Alerted} =
  VAR rd: Rd.T;
  BEGIN
    rd := Rsrc.Open (name, path);
    TRY RETURN InitFromRd (fv, rd, raw, path) FINALLY Rd.Close (rd) END
  END InitFromRsrc;

PROCEDURE InitFromSx (fv         : T;
                      description: S_exp;
                      raw                    := FALSE;
                      path       : Rsrc.Path := NIL    ): T RAISES {Error} =
  VAR state := cleanState;
  BEGIN
    fv.getVBT := TxtRefTbl.New ();
    fv.mu := NEW (MUTEX);
    fv.keyRec := NEW (REF VBT.KeyRec);
    fv.mouseRec := NEW (REF VBT.MouseRec);
    fv.positionRec := NEW (REF VBT.PositionRec);
    fv.miscRec := NEW (REF VBT.MiscRec);
    fv.path := path;
    IF raw THEN                 (* fv = (Filter parsedVBT) *)
      fv.raw := TRUE;
      EVAL Filter.T.init (fv, Parse (fv, description, state))
    ELSE
      (* fv = (Filter (ZSplit (MenuBar (Highlight (Clipboard (Reactivity
         parsedVBT)))))) *)
      fv.raw := FALSE;
      (* The trick here is that state.zsplit must already be set BEFORE we
         parse the description. *)
      WITH react     = NEW (FVFilter),
           clip      = NEW (ClipboardVBT.T).init (react),
           highlight = NEW (HighlightVBT.T).init (clip),
           menubar   = NEW (FVMenuBar).init (highlight),
           zsplit    = NEW (ZSplit.T).init (menubar)      DO
        EVAL Filter.T.init (fv, zsplit);
        state.zsplit := zsplit;
        state.menubar := menubar;
        EVAL react.init (Parse (fv, description, state))
      END
    END;
    RETURN fv
  END InitFromSx;

PROCEDURE GetZSplit (fv: T): ZSplit.T RAISES {Error} =
  BEGIN
    IF fv.raw THEN RAISE Error ("Uncooked FormsVBT (GetZSplit)") END;
    RETURN Filter.Child (fv)
  END GetZSplit;

PROCEDURE Insert (fv         : T;
                  parent     : TEXT;
                  description: TEXT;
                  at         : CARDINAL := LAST (CARDINAL)): VBT.T
  RAISES {Error} =
  VAR
    stateRef: REF State := VBT.GetProp (GetVBT (fv, parent),
                                        TYPECODE (REF State));
    res: VBT.T;
  BEGIN
    TRY
      res := Parse (fv, Sx.FromText (description, syntax := FVSyntax),
               stateRef^);
      InsertVBT (fv, parent, res, at);
      RETURN res
    EXCEPT
    | Sx.ReadError (txt) => RAISE Error ("Sx.ReadError: " & txt)
    | Rd.EndOfFile => RAISE Error ("End of input")
    | Thread.Alerted => RAISE Error ("Thread.Alerted")
    END
  END Insert;
  
PROCEDURE InsertFromFile (fv      : T;
                          parent  : TEXT;
                          filename: TEXT;
                          at      : CARDINAL := LAST (CARDINAL)): VBT.T
  RAISES {Error, Rd.Failure, Thread.Alerted} =
  VAR
    pathname: TEXT;
    rd      : Rd.T;
  BEGIN
    TRY
      pathname := Filename.ExpandTilde (filename);
      rd := FileStream.OpenRead (pathname);
      TRY
        RETURN Insert (fv, parent, Rd.GetText (rd, LAST (CARDINAL)), at)
      FINALLY
        Rd.Close (rd)
      END
    EXCEPT
    | Filename.Error => RAISE Error ("Bad filename: " & filename)
    END
  END InsertFromFile;

PROCEDURE InsertFromRsrc (fv    : T;
                          parent: TEXT;
                          name  : TEXT;
                          path  : Rsrc.Path;
                          n     : CARDINAL    := LAST (CARDINAL)): VBT.T
  RAISES {Error, Rd.Failure, Rsrc.NotFound, Thread.Alerted} =
  BEGIN
    RETURN Insert (fv, parent, Rsrc.Get (name, path), n)
  END InsertFromRsrc;

(*************************** snapshots *******************************)

PROCEDURE GetVal (fv: T; name: TEXT): REFANY =
  (* Returns value of name as REFANY, if a value can be
     retrieved *)
  BEGIN
    TRY
      WITH ri = NEW(REF INTEGER) DO
        ri^ := GetInteger(fv, name);
        RETURN ri
      END
    EXCEPT
      Error, Unimplemented =>
    END;
    TRY
      WITH rb = NEW(REF BOOLEAN) DO
        rb^ := GetBoolean(fv, name);
        RETURN rb
      END
    EXCEPT
      Error, Unimplemented =>
    END;
    TRY
      RETURN GetText(fv, name);
    EXCEPT
      Error, Unimplemented =>
    END;
    RETURN NIL
  END GetVal;

PROCEDURE Snapshot (fv: T; wr: Wr.T) RAISES {Error} =
  VAR
    key        : TEXT;
    val        : REFANY;
    keys, alist: List.T := NIL;
  BEGIN
    TRY
      keys := fv.getVBT.toKeyList ();
      WHILE keys # NIL DO
        key := keys.first;
        val := GetVal (fv, key);
        IF val # NIL THEN
          List.Push (
            alist, List.List2 (SxSymbol.FromName (key), val))
        END;
        keys := keys.tail
      END;
      Sx.Print (wr, alist);
      Wr.PutChar (wr, '\n')
    EXCEPT
      Sx.PrintError, Thread.Alerted, Wr.Failure =>
        RAISE Error ("Problem writing snapshot");
    END
  END Snapshot;

PROCEDURE Restore (fv: T; rd: Rd.T) RAISES {Mismatch, Error} =
  VAR
    refSx    : S_exp;
    sx       : List.T;
    mismatch : BOOLEAN;
    ignoreRef: REFANY;
  BEGIN
    TRY
      refSx := Sx.Read (rd);
      IF refSx = NIL OR NOT ISTYPE (refSx, List.T) THEN
        RAISE Error ("Snapshot is not a valid s-expression")
      END;
      sx := refSx;
      mismatch := FALSE;
      WHILE sx # NIL DO
        TYPECASE List.First (sx) OF
        | List.T (l) =>
            IF List.Length (l) # 2 THEN
              RAISE Error ("Illegal expression in snapshot")
            END;
            TYPECASE List.First (l) OF
            | SxSymbol.T (sym) =>
                IF NOT fv.getVBT.in (sym.name, ignoreRef) THEN
                  mismatch := TRUE
                ELSE
                  TYPECASE List.Second (l) OF
                  | TEXT (text) => PutText (fv, sym.name, text)
                  | REF BOOLEAN (refBool) =>
                      PutBoolean (fv, sym.name, refBool^)
                  | REF INTEGER (refInt) => PutInteger (fv, sym.name, refInt^)
                  ELSE
                    RAISE Error ("Value of component " & sym.name
                                   & " has illegal type");
                  END
                END
            ELSE
              RAISE Error ("Illegal component name in snapshot");
            END
        ELSE
          RAISE Error ("Snapshot is not a valid s-expression");
        END;
        sx := List.Tail (sx);
      END;
      IF mismatch THEN RAISE Mismatch END;
    EXCEPT
      Sx.ReadError, Rd.EndOfFile, Rd.Failure, Thread.Alerted, Unimplemented =>
        RAISE Error ("Problem with reading snapshot")
    END;
  END Restore;


(* ========================= Attachment ========================= *)

TYPE ClosureRef = BRANDED REF RECORD fv: T; name: TEXT; cl: Closure END;

PROCEDURE Attach (fv: T; name: TEXT; cl: Closure)
  RAISES {Error} =
  VAR vbt := GetVBT (fv, name);
  BEGIN
    TYPECASE vbt OF
    | FVTextEdit, FVTypeIn, FVTextArea, FVTypescript =>
        (* Attach it to the TextPort *)
        vbt := TextEditVBT.GetPort (vbt)
    ELSE
    END;
    IF cl # NIL THEN
      VBT.PutProp (
        vbt, NEW (ClosureRef, fv := fv, name := name, cl := cl))
    ELSE
      VBT.RemProp (vbt, TYPECODE (ClosureRef))
    END
  END Attach;

PROCEDURE MouseProc (self: VBT.T; READONLY cd: VBT.MouseRec) =
  (*
     This is the callback for self =
         Boolean (BooleanVBT.T),
         Browser (ListVBT.T),
         Button (SwitchVBT.T),
         Guard (GuardedBtnVBT.T),
         MButton (MenuSwitchVBT.T),
         Menu (AnchorSplit.T 'pre' method),
         PopButton (SwitchVBT.T),
         PopMButton (MenuSwitchVBT.T),
         Radio,
         Scroller (NumericScrollerVBT.T),
         Source (SourceVBT.T), and
         TrillButton (TrillSwitchVBT.T)
         ZChassis (via CloseButton)
         ZChild (via CloseButton)
         *)
  VAR
    cr: ClosureRef := VBT.GetProp (self, TYPECODE (ClosureRef));
    fv: T;
  BEGIN
    IF cr # NIL THEN
      fv := cr.fv;
      LOCK fv.mu DO
        INC (fv.eventCount);
        IF fv.eventCount = 1 THEN
          fv.mouseRec^ := cd;
          fv.eventCode := TYPECODE (REF VBT.MouseRec)
        END
      END;
      TRY
        cr.cl.apply (fv, cr.name, cd.time)
      FINALLY
        LOCK fv.mu DO DEC (fv.eventCount) END
      END;
    END
  END MouseProc;

TYPE
  OldClosure = Closure OBJECT
                 ref : REFANY;
                 proc: Proc
               OVERRIDES
                 apply := OldApply
               END;
                    
PROCEDURE AttachProc (fv: T; name: TEXT; proc: Proc; cl: REFANY := NIL)
  RAISES {Error} =
  BEGIN
    IF proc # NIL THEN
      Attach (fv, name, NEW (OldClosure, ref := cl, proc := proc))
    ELSE
      Attach (fv, name, NIL)
    END
  END AttachProc;
  
PROCEDURE OldApply (oc: OldClosure; fv: T; name: TEXT; time: VBT.TimeStamp) =
  BEGIN
    oc.proc (fv, name, oc.ref, time)
  END OldApply;

TYPE ReservedVBT = VBT.Leaf BRANDED OBJECT END;

PROCEDURE AddSymbol (fv: T; name: TEXT) RAISES {Error} =
  VAR ref: REFANY;
  BEGIN
    LOCK fv.mu DO
      IF fv.getVBT.in (name, ref) THEN
        RAISE Error ("The name " & name & " is already in use.")
      ELSE
        EVAL fv.getVBT.put (name, NEW (ReservedVBT))
      END
    END
  END AddSymbol;

PROCEDURE AddUniqueSymbol (fv: T): TEXT =
  VAR
    ref : REFANY;
    name: TEXT;
  BEGIN
    LOCK fv.mu DO
      LOOP
        name := "-v-b-t-" & Fmt.Int (fv.gensym);
        IF fv.getVBT.in (name, ref) THEN INC (fv.gensym) ELSE EXIT END
      END;
      EVAL fv.getVBT.put (name, NEW (ReservedVBT));
      RETURN name
    END
  END AddUniqueSymbol;


(* ===================== MakeEvent & GetTheEvent ==================== *)

VAR (* CONST *) MakeEventSelection: VBT.Selection;

PROCEDURE MakeEvent (fv: T; name: TEXT; time: VBT.TimeStamp) RAISES {Error} =
  <* LL = VBT.mu *>
  VAR
    cr : ClosureRef;
    vbt             := GetVBT (fv, name);
  BEGIN
    TYPECASE vbt OF
      FVTextEdit, FVTypeIn, FVTextArea, FVTypescript => 
        vbt := TextEditVBT.GetPort (vbt)
    ELSE
    END;
    cr := VBT.GetProp (vbt, TYPECODE (ClosureRef));
    IF cr = NIL THEN RAISE Error ("Nothing attached to " & name) END;
    LOCK fv.mu DO
      INC (fv.eventCount);
      IF fv.eventCount = 1 THEN
        fv.miscRec.type := MakeEventMiscCodeType;
        fv.miscRec.time := time;
        fv.miscRec.selection := MakeEventSelection;
        fv.eventCode := TYPECODE (REF VBT.MiscRec)
      END
    END;
    TRY
      cr.cl.apply (cr.fv, cr.name, time)
    FINALLY
      LOCK fv.mu DO DEC (fv.eventCount) END
    END
  END MakeEvent;

PROCEDURE GetTheEvent (fv: T): AnyEvent.Code RAISES {Error} =
  VAR tc: CARDINAL;
  BEGIN
    LOCK fv.mu DO
      tc := fv.eventCode;
      IF fv.eventCount = 0 THEN
        RAISE Error ("There is no active event")
      (*
      ELSIF fv.eventCount > 1 THEN
        RAISE Error ("More than 1 event is active")
      *)
      ELSIF tc = TYPECODE (REF VBT.KeyRec) THEN
        RETURN AnyEvent.KeyToCode (fv.keyRec^)
      ELSIF tc = TYPECODE (REF VBT.MouseRec) THEN
        RETURN AnyEvent.MouseToCode (fv.mouseRec^)
      ELSIF tc = TYPECODE (REF VBT.PositionRec) THEN
        RETURN AnyEvent.PositionToCode (fv.positionRec^)
      ELSIF tc = TYPECODE (REF VBT.MiscRec) THEN
        RETURN AnyEvent.MiscToCode (fv.miscRec^)
      ELSE
        RAISE
          Error ("Internal error: The active event has an unknown type")
      END
    END
  END GetTheEvent;

PROCEDURE GetTheEventTime (fv: T): VBT.TimeStamp RAISES {Error} =
  BEGIN                         (* Very lazy. *)
    RETURN AnyEvent.TimeStamp (GetTheEvent (fv))
  END GetTheEventTime;

(************************ Text-widget callback ************************)

REVEAL
  Port = TextPort.T BRANDED OBJECT
         OVERRIDES
           returnAction := DeliverText;
           focus        := Focus;
           filter       := FilterProc
         END;

PROCEDURE DeliverText (fvport: TextPort.T; READONLY cd: VBT.KeyRec) =
  (* Callback for our TextPorts.  Same as MouseProc, except that we get a
     KeyRec, not a MouseRec. *)
  VAR
    cr: ClosureRef := VBT.GetProp (fvport, TYPECODE (ClosureRef));
    fv: T;
  BEGIN
    IF cr # NIL THEN
      fv := cr.fv;
      LOCK fv.mu DO
        INC (fv.eventCount);
        IF fv.eventCount = 1 THEN
          fv.keyRec^ := cd;
          fv.eventCode := TYPECODE (REF VBT.KeyRec)
        END
      END;
      TRY
        cr.cl.apply (cr.fv, cr.name, cd.time)
      FINALLY
        LOCK fv.mu DO DEC (fv.eventCount) END
      END
    ELSE
      TYPECASE fvport OF
      | TypescriptPort (x) => TypescriptVBT.Port.returnAction (x, cd)
      ELSE
        TextPort.T.returnAction (fvport, cd)
      END
    END;
  END DeliverText;
  
REVEAL
  TypescriptPort = TypescriptVBT.Port BRANDED OBJECT
                   OVERRIDES
                     returnAction := DeliverText; (* was -ToBoth *)
                     focus        := Focus;
                     filter       := FilterProc
                   END;

(* Don't do this.  If you don't attach a procedure to the typescript,
   this will cause newlines to added twice to the textport, once
   via TextPort.T.returnAction and once via TypescriptVBT.Port.returnAction.
   Thanks to Dave Nichols for spotting this.
PROCEDURE DeliverTextToBoth (         v : TypescriptPort;
                             READONLY cd: VBT.KeyRec      ) =
  BEGIN
    TypescriptVBT.Port.returnAction (v, cd);
    DeliverText (v, cd)
  END DeliverTextToBoth;
*)
  
TYPE
  FilterClosureRef = REF RECORD
                           fv    : T;
                           name  : TEXT;
                           filter: KeyFilter
                         END;

PROCEDURE AttachKeyFilter (fv: T; name: TEXT; filter: KeyFilter)
  RAISES {Error} =
  (* This attaches a callback for the 'filter' method of the TextPort. *)
  VAR vbt := GetVBT (fv, name);
  BEGIN
    TYPECASE vbt OF
    | FVTextEdit, FVTypeIn, FVTextArea =>
        IF filter # NIL THEN
          VBT.PutProp (
            TextEditVBT.GetPort (vbt),
            NEW (FilterClosureRef, fv := fv, name := name, filter := filter))
        ELSE
          VBT.RemProp (TextEditVBT.GetPort (vbt), TYPECODE (FilterClosureRef))
        END
    ELSE
      RAISE Error (name & " isn't a text-edit interactor")
    END
  END AttachKeyFilter;

PROCEDURE FilterProc (port: TextPort.T; VAR cd: VBT.KeyRec) =
  VAR
    cl: FilterClosureRef := VBT.GetProp (port, TYPECODE (FilterClosureRef));
    fv: T;
  BEGIN
    IF cl # NIL THEN
      fv := cl.fv;
      LOCK fv.mu DO
        INC (fv.eventCount);
        IF fv.eventCount = 1 THEN
          fv.keyRec^ := cd;
          fv.eventCode := TYPECODE (REF VBT.KeyRec)
        END
      END;
      TRY
        cl.filter.apply (cl.fv, cl.name, cd)
      FINALLY
        LOCK fv.mu DO DEC (fv.eventCount) END
      END
    END
  END FilterProc;

TYPE
  FocusClosureRef = REF RECORD
                          fv   : T;
                          name : TEXT;
                          alert: FocusAlert
                        END;

PROCEDURE AttachFocusAlert (fv: T; name: TEXT; alert: FocusAlert)
  RAISES {Error} =
  (* This attaches a callback for the 'focus' method of the TextPort. *)
  VAR vbt := GetVBT (fv, name);
  BEGIN
    TYPECASE vbt OF
    | FVTextEdit, FVTypeIn, FVTextArea =>
        IF alert # NIL THEN
          VBT.PutProp (
            TextEditVBT.GetPort (vbt),
            NEW (FocusClosureRef, fv := fv, name := name, alert := alert))
        ELSE
          VBT.RemProp (TextEditVBT.GetPort (vbt), TYPECODE (FocusClosureRef))
        END
    ELSE
      RAISE Error (name & " isn't a text-edit interactor")
    END
  END AttachFocusAlert;

VAR (* CONST *) FocusSelection: VBT.Selection;
  
PROCEDURE Focus (port: TextPort.T; gaining: BOOLEAN; time: VBT.TimeStamp) =
  VAR
    cl: FocusClosureRef := VBT.GetProp (port, TYPECODE (FocusClosureRef));
    fv: T;
  BEGIN
    IF cl # NIL THEN
      fv := cl.fv;
      LOCK fv.mu DO
        INC (fv.eventCount);
        IF fv.eventCount = 1 THEN
          fv.miscRec.type := FocusMiscCodeType;
          fv.miscRec.detail [0] := ORD (gaining);
          fv.miscRec.time := time;
          fv.miscRec.selection := FocusSelection;
          fv.eventCode := TYPECODE (REF VBT.MiscRec)
        END
      END;
      TRY
        cl.alert.apply (cl.fv, cl.name, gaining, time)
      FINALLY
        LOCK fv.mu DO DEC (fv.eventCount) END
      END
    END
  END Focus;


(* ====================== FileBrowser ===================== *)

REVEAL 
  FVFileBrowser = FileBrowserVBT.T BRANDED OBJECT
    OVERRIDES
      activateFile := ActivateFileB;
    END;

PROCEDURE ActivateFileB (             self    : FVFileBrowser;
                         <* UNUSED *> filename: TEXT;
                                      event   : AnyEvent.Code     ) =
  (* callback for our FileBrowserVBTs. *)
  VAR mr: VBT.MouseRec;
  BEGIN
    TYPECASE event OF
    | AnyEvent.KeyCode (key) =>
        mr.time := key.key.time;
        MouseProc (self, mr);
    | AnyEvent.MouseCode (mouse) => MouseProc (self, mouse.mouse)
    ELSE
    END;
  END ActivateFileB;


(* ====================== Browser ===================== *)

REVEAL
  UniSelector = PrivateUniSelector BRANDED OBJECT
                OVERRIDES
                  insideClick := InsideClick
                END;

PROCEDURE InsideClick (v: UniSelector; cd: VBT.MouseRec; this: ListVBT.Cell) =
  BEGIN
    ListVBT.UniSelector.insideClick (v, cd, this);
    IF cd.clickType = VBT.ClickType.LastUp
         AND (v.quick OR cd.clickCount = 3) THEN
      MouseProc (v.browser, cd)
    END
  END InsideClick;

REVEAL
  MultiSelector = PrivateMultiSelector BRANDED OBJECT
                  OVERRIDES
                    insideClick := MultiInsideClick
                  END;

PROCEDURE MultiInsideClick (v   : MultiSelector;
                            cd  : VBT.MouseRec;
                            this: ListVBT.Cell   ) =
  BEGIN
    ListVBT.MultiSelector.insideClick (v, cd, this);
    IF cd.clickType = VBT.ClickType.LastUp
         AND (v.quick OR cd.clickCount = 3) THEN
      MouseProc (v.browser, cd)
    END
  END MultiInsideClick;

(* ====================== Buttons ===================== *)

REVEAL
  FVBoolean = BooleanVBT.T BRANDED OBJECT
              OVERRIDES
                callback := MouseProc;
              END;

REVEAL
  FVButton = SwitchVBT.T BRANDED OBJECT
             OVERRIDES
               callback := MouseProc;
             END;

REVEAL
  FVGuard = GuardedBtnVBT.T BRANDED OBJECT
            OVERRIDES
              callback := MouseProc;
            END;

REVEAL
  FVMButton = MenuSwitchVBT.T BRANDED OBJECT
              OVERRIDES
                callback := MouseProc;
              END;

REVEAL
  FVScroller = NumericScrollerVBT.T BRANDED OBJECT
               OVERRIDES
                 callback := MouseProc;
               END;

TYPE
  PublicFVSource =
    SourceVBT.T OBJECT OVERRIDES hit := SourceVBT.AlwaysHit END;

REVEAL
  FVSource =
    PublicFVSource BRANDED OBJECT OVERRIDES callback := MouseProc; END;

REVEAL
  FVTrillButton =
    TrillSwitchVBT.T BRANDED OBJECT OVERRIDES callback := MouseProc; END;

REVEAL
  FVZChassis =
    ZChassisVBT.T BRANDED OBJECT OVERRIDES callback := MouseProc; END;

(* ====================== Radio & Choice ===================== *)

REVEAL
  FVChoice =
    PrivateChoice BRANDED OBJECT OVERRIDES callback := ChoiceCallback; END;

PROCEDURE ChoiceCallback (self: FVChoice; READONLY cd: VBT.MouseRec) =
  BEGIN
    MouseProc (self, cd);
    MouseProc (self.radio, cd)
  END ChoiceCallback;


(* ====================== PopButton & PopMButton ===================== *)

REVEAL
  FVPopButton = SwitchVBT.T BRANDED OBJECT
                OVERRIDES
                  callback := PopButtonProc;
                END;

REVEAL
  FVPopMButton = MenuSwitchVBT.T BRANDED OBJECT
                 OVERRIDES
                   callback := PopButtonProc;
                 END;
    
TYPE PopTarget = BRANDED REF ZChildVBT.T;

PROCEDURE PopButtonProc (self: VBT.T; READONLY cd: VBT.MouseRec) =
  (* Callback procedure for Pop[M]Button *)
  VAR
    popTarget: PopTarget := VBT.GetProp (self, TYPECODE (PopTarget));
    zchild   : VBT.T;
  BEGIN
    IF popTarget # NIL THEN
      zchild := ZSplitUtils.FindZChild (popTarget^);
      IF zchild # NIL THEN ZChildVBT.Pop (zchild) END
    END;
    MouseProc (self, cd)
  END PopButtonProc;

PROCEDURE SetPopTarget (source: ButtonVBT.T; target: ZChildVBT.T) =
  BEGIN
    WITH prop = NEW (PopTarget) DO
      prop^ := target;
      VBT.PutProp (source, prop)
    END
  END SetPopTarget;


(* =========================== PageButton ============================ *)

REVEAL
  FVPageButton = PrivatePageButton BRANDED OBJECT
                   backwards := FALSE
                 OVERRIDES
                   callback := PageButtonProc;
                   init     := InitPageButton
                 END;

PROCEDURE InitPageButton (b        : FVPageButton;
                          ch       : VBT.T;
                          shadow   : Shadow.T;
                          backwards: BOOLEAN     ): FVPageButton =
  BEGIN
    EVAL
      SwitchVBT.T.init (b, NEW (ShadowedFeedbackVBT.T).init (ch, shadow));
    b.backwards := backwards;
    RETURN b
  END InitPageButton;

PROCEDURE PageButtonProc (self: VBT.T; READONLY cd: VBT.MouseRec) =
  VAR
    pb           : FVPageButton := self;
    tsplit                    := pb.target;
    current, next: VBT.T;
  <* FATAL Split.NotAChild *>
  BEGIN
    current := TSplit.GetCurrent (tsplit);
    IF pb.backwards THEN
      next := Split.Pred (tsplit, current);
      IF next = NIL AND tsplit.circular THEN
        next := Split.Pred (tsplit, NIL)
      END
    ELSE
      next := Split.Succ (tsplit, current);
      IF next = NIL AND tsplit.circular THEN
        next := Split.Succ (tsplit, NIL)
      END
    END;
    IF next # NIL THEN TSplit.SetCurrent (tsplit, next) END;
    MouseProc (self, cd)
  END PageButtonProc;


(* =========================== LinkButton ============================ *)

REVEAL
  FVLinkButton = PrivateLinkButton BRANDED OBJECT
               OVERRIDES
                 callback := LinkButtonProc;
                 init     := InitLinkButton
               END;

PROCEDURE InitLinkButton (b        : FVLinkButton;
                          ch       : VBT.T;
                          shadow   : Shadow.T): FVLinkButton =
  BEGIN
    EVAL
      SwitchVBT.T.init (b, NEW (ShadowedFeedbackVBT.T).init (ch, shadow));
    RETURN b
  END InitLinkButton;

PROCEDURE LinkButtonProc (self: VBT.T; READONLY cd: VBT.MouseRec) =
  VAR lb: FVLinkButton := self;
  BEGIN
    TRY
      TSplit.SetCurrent (lb.Tparent, lb.Tchild)
    EXCEPT
      Split.NotAChild => (* ignore *)
    END;
    MouseProc (self, cd)
  END LinkButtonProc;


(* =========================== CloseButton ============================ *)

REVEAL
  FVCloseButton = PrivateCloseButton BRANDED OBJECT
                  OVERRIDES
                    callback := CloseButtonProc;
                    init     := InitCloseButton
                  END;

PROCEDURE InitCloseButton (b: FVCloseButton; ch: VBT.T; shadow: Shadow.T):
  FVCloseButton =
  BEGIN
    EVAL
      SwitchVBT.T.init (b, NEW (ShadowedFeedbackVBT.T).init (ch, shadow));
    RETURN b
  END InitCloseButton;

PROCEDURE CloseButtonProc (         self: FVCloseButton;
                           READONLY cd  : VBT.MouseRec   ) =
  VAR zch := ZSplitUtils.FindZChild (self.target);
  BEGIN
    IF zch # NIL THEN
      ZSplit.Unmap (zch);
      MouseProc (self, cd);
      MouseProc (zch, cd)
    END
  END CloseButtonProc;


(* ============================= HBox, VBox ============================== *)

REVEAL
  FVHBox = HVSplit.T BRANDED OBJECT
           OVERRIDES
             shape := HVSplitShape;
           END;

REVEAL
  FVVBox = HVSplit.T BRANDED OBJECT
           OVERRIDES
             shape := HVSplitShape;
           END;

CONST
  EmptyShape = VBT.SizeRange{lo := 0, pref := 0, hi := 1};

PROCEDURE HVSplitShape(v: HVSplit.T; ax: Axis.T; n: CARDINAL): 
    VBT.SizeRange RAISES {} =
  BEGIN
    IF v.succ(NIL) = NIL THEN 
      RETURN EmptyShape
    ELSE
      RETURN HVSplit.T.shape(v, ax, n)
    END
  END HVSplitShape;
        
    
(* ============================= HTile, VTile ============================== *)

REVEAL
  FVHTile =
    SplitterVBT.T BRANDED OBJECT OVERRIDES shape := HVTileShape; END;

REVEAL
  FVVTile =
    SplitterVBT.T BRANDED OBJECT OVERRIDES shape := HVTileShape; END;

PROCEDURE HVTileShape(v: SplitterVBT.T; ax: Axis.T; n: CARDINAL): 
    VBT.SizeRange RAISES {} =
  BEGIN
    IF v.succ(NIL) = NIL THEN 
      RETURN EmptyShape
    ELSE
      RETURN SplitterVBT.T.shape(v, ax, n)
    END
  END HVTileShape;
        
    
(* ============================= Numeric ============================== *)

REVEAL
  FVNumeric =
    NumericVBT.T BRANDED OBJECT OVERRIDES callback := NumericProc; END;
    
PROCEDURE NumericProc (self: FVNumeric; event: AnyEvent.Code) =
  (* Callback procedure for Numeric (NumericVBT.T) *)
  VAR mr: VBT.MouseRec;
  BEGIN
    TYPECASE event OF
    | AnyEvent.KeyCode (key) => mr.time := key.key.time; MouseProc (self, mr);
    | AnyEvent.MouseCode (mouse) => MouseProc (self, mouse.mouse)
    ELSE
    END;
  END NumericProc;


(* ============================= Menu ============================== *)

REVEAL
  FVMenu = AnchorSplit.T BRANDED OBJECT 
    OVERRIDES
      pre := PreMenu;
    END;

PROCEDURE PreMenu (v: AnchorSplit.T) =
  VAR mouse: VBT.MouseRec;
  BEGIN
    mouse.time := 0;
    MouseProc (v, mouse);
    AnchorSplit.T.pre (v);
  END PreMenu;


(* ====================== Runtime support routines ==================== *)

PROCEDURE GetText (fv: T; name: TEXT): TEXT RAISES {Error, Unimplemented} =
  VAR v := GetVBT (fv, name);
  BEGIN
    TYPECASE v OF
    | FVFileBrowser (v) =>
        TRY
          RETURN FileBrowserVBT.GetFile (v)
        EXCEPT
        | FileBrowserVBT.Error (e) =>
            RAISE Error (Fmt.F ("Error for %s: %s", e.path, e.text))
        END
    | FVText (t) => RETURN TextVBT.Get (t)
    | FVTypescript (v) =>
        TRY
          RETURN Rd.GetText (TypescriptVBT.GetRd (v), LAST (CARDINAL))
        EXCEPT
        | Rd.Failure (ref) => RAISE Error (RdUtils.FailureText (ref))
        | Thread.Alerted => RAISE Error ("Thread.Alerted")
        END
    | FVTextEdit, FVTypeIn, FVTextArea =>
        RETURN TextPort.GetText (TextEditVBT.GetPort (v))
    | FVNumeric (v) =>
        IF NumericVBT.IsEmpty (v) THEN
          RETURN ""
        ELSE
          RETURN Fmt.Int (NumericVBT.Get (v))
        END
    ELSE
      RAISE Unimplemented
    END
  END GetText;

PROCEDURE PutText (fv: T; name: TEXT; text: TEXT; append := FALSE)
  RAISES {Error, Unimplemented} =
  VAR v := GetVBT (fv, name);
  BEGIN
    TYPECASE v OF
    | FVFileBrowser (v) =>
        TRY
          FileBrowserVBT.Set (v, text);
          RETURN
        EXCEPT
          FileBrowserVBT.Error (e) =>
            RAISE
              Error (Fmt.F ("Error for %s: %s", e.path, e.text))
        END
    | FVText (t) =>
        IF append THEN
          TextVBT.Put (t, TextVBT.Get (t) & text)
        ELSE
          TextVBT.Put (t, text)
        END
    | FVTypescript (v) =>
        TRY
          Wr.PutText (TypescriptVBT.GetWr (v), text)
        EXCEPT
        | Wr.Failure (ref) =>
            RAISE Error (RdUtils.FailureText (ref))
        | Thread.Alerted =>     (* ignore *)
        END
    | FVTextEdit, FVTypeIn, FVTextArea =>
        IF append THEN
          TextPort.PutText (TextEditVBT.GetPort (v), text)
        ELSE
          TextPort.SetText (TextEditVBT.GetPort (v), text)
        END
    ELSE
      RAISE Unimplemented
    END
  END PutText;

PROCEDURE GetInteger (fv: T; name: TEXT): INTEGER
  RAISES {Error, Unimplemented} =
  <* FATAL Split.NotAChild *>
  BEGIN
    TYPECASE GetVBT (fv, name) OF
    | FVScroller (t) => RETURN NumericScrollerVBT.Get (t)
    | FVNumeric (t) => RETURN NumericVBT.Get (t)
    | FVTSplit (t) =>
        RETURN Split.Index (t, TSplit.GetCurrent (t))
    ELSE
      RAISE Unimplemented
    END
  END GetInteger;

PROCEDURE PutInteger (fv: T; name: TEXT; int: INTEGER)
  RAISES {Error, Unimplemented} =
  <* FATAL Split.NotAChild *>
  BEGIN
    TYPECASE GetVBT (fv, name) OF
    | FVScroller (t) => NumericScrollerVBT.Put (t, int); RETURN
    | FVNumeric (t) => NumericVBT.Put (t, int); RETURN
    | FVTSplit (t) =>
        IF 0 <= int AND int < Split.NumChildren (t) THEN
          TSplit.SetCurrent (t, Split.Nth (t, int));
          RETURN
        ELSE
          RAISE Error (
                  Fmt.F ("%s is an illegal TSplit-index for %s.",
                         Fmt.Int (int), name))
        END
    ELSE
      RAISE Unimplemented
    END
  END PutInteger;

PROCEDURE PutBoolean (fv: T; name: TEXT; val: BOOLEAN)
  RAISES {Error, Unimplemented} =
  BEGIN
    TYPECASE GetVBT (fv, name) OF
    | FVBoolean (b) => BooleanVBT.Put (b, val); RETURN
    | FVChoice (c) =>
        IF val THEN
          ChoiceVBT.Put (c)
        ELSIF ChoiceVBT.Get (c) = c THEN
          ChoiceVBT.Clear (c)
        END;
        RETURN
    ELSE
      RAISE Unimplemented
    END
  END PutBoolean;

PROCEDURE PutChoice (fv: T; radioName, choiceName: TEXT)
  RAISES {Error, Unimplemented} =
  BEGIN
    TYPECASE GetVBT (fv, radioName) OF
    | FVRadio (r) =>
        IF choiceName = NIL THEN
          WITH cur = Radio.Get (r.radio) DO
            IF cur # NIL THEN ChoiceVBT.Clear (cur); RETURN END
          END
        ELSE
          TYPECASE GetVBT (fv, choiceName) OF
          | FVChoice (c) => ChoiceVBT.Put (c); RETURN
          ELSE
            RAISE Error ("No Choice named " & choiceName)
          END
        END
    ELSE
      RAISE Unimplemented
    END
  END PutChoice;

(************************  Direct access  *************************)

PROCEDURE SetVBT (fv: T; name: TEXT; vbt: VBT.T) RAISES {Error} =
  BEGIN
    LOCK fv.mu DO
      IF fv.getVBT.put (name, vbt) THEN
        RAISE Error ("There is already a VBT named " & name)
      END
    END
  END SetVBT;

PROCEDURE GetVBT (fv: T; name: TEXT): VBT.T RAISES {Error} =
  VAR result: REFANY;
  BEGIN
    LOCK fv.mu DO
      IF fv.getVBT.in (name, result) THEN RETURN result END;
      RAISE Error ("There is no VBT named " & name)
    END
  END GetVBT;

PROCEDURE Delete (fv: T; parent: TEXT; at: CARDINAL; count: CARDINAL := 1)
  RAISES {Error} =
  BEGIN
    TRY
      WITH p = GetVBT (fv, parent), at = MIN (at, Multi.NumChildren (p)) DO
        FOR i := 1 TO MIN (count, Multi.NumChildren (p) - at) DO
          Multi.Delete (p, Multi.Nth (p, at))
        END;
        RETURN
      END
    EXCEPT
      Multi.NotAChild => RAISE Error ("Delete: No Split named " & parent)
    END
  END Delete;

PROCEDURE InsertVBT (fvLocal: T;
                     parent : TEXT;
                     child  : VBT.T;
                     at     : CARDINAL := LAST (CARDINAL)) RAISES {Error} =
  BEGIN
    TRY
      WITH
        p = GetVBT (fvLocal, parent),
        at = MIN (at, Multi.NumChildren (p)) DO
        IF at = 0 THEN
          Multi.Insert (p, NIL, child)
        ELSE
          Multi.Insert (p, Multi.Nth (p, at - 1), child)
        END
      END
    EXCEPT
      Multi.NotAChild =>
        RAISE Error ("InsertVBT: No Split named " & parent)
    END
  END InsertVBT;

(***********************  Special controls  ***********************)

PROCEDURE TakeFocus (fv       : T;
                     name     : TEXT;
                     eventTime: VBT.TimeStamp;
                     select                     := FALSE) RAISES {Error} =
  VAR vbt := GetVBT (fv, name);
  PROCEDURE focus (port: TextPort.T) =
    BEGIN
      IF TextPort.TryFocus (port, eventTime) AND select THEN
        TextPort.Select (
          port, eventTime, 0, LAST (CARDINAL), replaceMode := TRUE)
      END
    END focus;
  BEGIN
    TYPECASE vbt OF
    | FVHelper (h) => focus (h)
    | FVTextEdit, FVTypeIn, FVTextArea => focus (TextEditVBT.GetPort (vbt))
    | FVNumeric (v) => focus (NumericVBT.GetTypein (v))
    ELSE
      RAISE Error (name & " cannot take a keyboard focus")
    END
  END TakeFocus;


(************************ Pop Up and Down ************************)

PROCEDURE PopUp (fv        : T;
                 name      : TEXT;
                 <* UNUSED *> eventTime : VBT.TimeStamp;
                 forcePlace: BOOLEAN         := FALSE) RAISES {Error} =
  BEGIN
    ZChildVBT.Pop(FindZChildAncestor(fv, name), forcePlace)
  END PopUp;

PROCEDURE PopDown (fv: T; name: TEXT) RAISES {Error} =
  BEGIN
    ZSplit.Unmap(FindZChildAncestor(fv, name))
  END PopDown;

PROCEDURE FindZChildAncestor (fv: T; name: TEXT): VBT.T RAISES {Error} =
  VAR
    target := GetVBT (fv, name);
    zchild := ZSplitUtils.FindZChild (target);
  BEGIN
    IF zchild = NIL THEN
      RAISE Error ("VBT named " & name
                     & "isn't in a non-background child of a ZSplit");
    END;
    RETURN zchild
  END FindZChildAncestor;


(************************ Runtime properties ************************)

PROCEDURE GetTextProperty (fv: T; name, prop: TEXT): TEXT
  RAISES {Error, Unimplemented} =
  VAR
    vbt                 := GetVBT (fv, name);
    stateRef: REF State := VBT.GetProp (vbt, TYPECODE (REF State));
  BEGIN
    IF stateRef = NIL THEN
      RAISE Error (Fmt.F ("The form named \"%s\" has no properties", name))
    ELSIF Text.Equal (prop, "BgColor") THEN
      RETURN ColorName.FromRGB (stateRef.bgRGB) (* Cache this!! *)
    ELSIF Text.Equal (prop, "Color") THEN
      RETURN ColorName.FromRGB (stateRef.fgRGB)
    ELSIF Text.Equal (prop, "DarkShadow") THEN
      RETURN ColorName.FromRGB (stateRef.darkRGB)
    ELSIF Text.Equal (prop, "LightShadow") THEN
      RETURN ColorName.FromRGB (stateRef.lightRGB)
    ELSIF Text.Equal (prop, "Font") THEN
      RETURN stateRef.fontName
    ELSIF Text.Equal (prop, "LabelFont") THEN
      RETURN stateRef.labelFontName
    ELSE
      RAISE Unimplemented
    END
  END GetTextProperty;
                       
PROCEDURE PutTextProperty (fv: T; name, property: TEXT; t: TEXT)
  RAISES {Error, Unimplemented} =
  VAR
    vbt                 := GetVBT (fv, name);
    stateRef: REF State := VBT.GetProp (vbt, TYPECODE (REF State));
  BEGIN
    IF stateRef = NIL THEN
      RAISE Error (Fmt.F ("The form named \"%s\" has no properties", name))
    ELSIF Text.Equal (property, "BgColor") THEN
      ChangeColor (t, stateRef, stateRef.bgRGB, stateRef.bgOp);
      TYPECASE vbt OF
      | FVGlue (v) => TextureVBT.Set (Filter.Child (v), stateRef.bgOp)
      | FVRim (v) => BorderedVBT.SetColor (v, stateRef.bgOp)
      | FVTextEdit, FVTypeIn, FVTextArea =>
          NARROW (vbt, TextEditVBT.T).setColorScheme (
            PaintOpCache.MakeColorScheme (stateRef.bgOp, stateRef.fgOp))
      | FVText (v) =>
          TextVBT.SetFont (
            v, TextVBT.GetFont (v),
            PaintOpCache.MakeColorQuad (stateRef.bgOp, stateRef.fgOp))
      ELSE
        RAISE Unimplemented
      END
    ELSIF Text.Equal (property, "Color") THEN
      ChangeColor (t, stateRef, stateRef.fgRGB, stateRef.fgOp);
      TYPECASE vbt OF
      | FVBar (v) => TextureVBT.Set (Filter.Child (v), stateRef.fgOp)
      | FVBorder (v) => BorderedVBT.SetColor (v, stateRef.fgOp)
      | FVTextEdit, FVTypeIn, FVTextArea =>
          NARROW (vbt, TextEditVBT.T).setColorScheme (
            PaintOpCache.MakeColorScheme (stateRef.bgOp, stateRef.fgOp))
      | FVText (v) =>
          TextVBT.SetFont (
            v, TextVBT.GetFont (v),
            PaintOpCache.MakeColorQuad (stateRef.bgOp, stateRef.fgOp))
      ELSE
        RAISE Unimplemented
      END
    ELSIF Text.Equal (property, "Font") THEN
      stateRef.fontName := t;
      stateRef.font := FindFont (t);
      stateRef.fontMetrics := NIL;
      TYPECASE vbt OF
      | FVTextEdit, FVTextArea =>
          NARROW (vbt, TextEditVBT.T).setFont (stateRef.font)
      ELSE
        RAISE Unimplemented
      END
    ELSIF Text.Equal (property, "LabelFont") THEN
      stateRef.labelFontName := t;
      stateRef.labelFont := FindFont (t);
      stateRef.labelFontMetrics := NIL;
      TYPECASE vbt OF
      | FVText (v) =>
          TextVBT.SetFont (v, stateRef.labelFont, TextVBT.GetQuad (v))
      ELSE
        RAISE Unimplemented
      END
    ELSE
      RAISE Unimplemented
    END
  END PutTextProperty;

PROCEDURE ChangeColor (    name    : TEXT;
                           stateRef: REF State;
                       VAR rgb     : RGB.T;
                       VAR op      : PaintOp.T  ) RAISES {Error} =
  BEGIN
    TRY
      rgb := ColorName.ToRGB (name);
      op := PaintOpCache.FromRGB (rgb, PaintOp.Mode.Accurate);
      stateRef.shadow :=
        Shadow.New (stateRef.shadowSz, stateRef.bgOp, stateRef.fgOp,
                    stateRef.lightOp, stateRef.darkOp)
    EXCEPT
    | ColorName.NotFound => RAISE Error ("No such color: " & name)
    END
  END ChangeColor;
                                   
VAR fontCache := TxtIntTbl.New ();
  
PROCEDURE FindFont (fontname: TEXT): Font.T =
  VAR fontnumber: INTEGER;
  BEGIN
    IF fontCache.in (fontname, fontnumber) THEN
      RETURN Font.T {fontnumber}
    ELSE
      WITH f = Font.FromName (ARRAY OF TEXT {fontname}) DO
        EVAL fontCache.put (fontname, f.fnt);
        RETURN f
      END
    END
  END FindFont;

PROCEDURE MakeActive(fv: T; name: Text.T) RAISES {Error} =
  BEGIN
     SetReactivity(fv, name, ReactivityVBT.State.Active);
  END MakeActive;

PROCEDURE MakePassive(fv: T; name: Text.T) RAISES {Error} =
  BEGIN
     SetReactivity(fv, name, ReactivityVBT.State.Passive);
  END MakePassive;
  
PROCEDURE MakeDormant(fv: T; name: Text.T) RAISES {Error} =
  BEGIN
     SetReactivity(fv, name, ReactivityVBT.State.Dormant);
  END MakeDormant;
  
PROCEDURE MakeVanish(fv: T; name: Text.T) RAISES {Error} =
  BEGIN
     SetReactivity(fv, name, ReactivityVBT.State.Vanish);
  END MakeVanish;

PROCEDURE SetReactivity(fv: T; name: Text.T; state: ReactivityVBT.State)
  RAISES {Error} =
  BEGIN
    ReactivityVBT.Set(FindReactivityVBT(fv, name), state);
  END SetReactivity;


PROCEDURE IsActive(fv: T; name: Text.T): BOOLEAN RAISES {Error} =
  BEGIN
    RETURN TestReactivity(fv, name, ReactivityVBT.State.Active);
  END IsActive;

PROCEDURE IsPassive(fv: T; name: Text.T): BOOLEAN RAISES {Error} =
  BEGIN
    RETURN TestReactivity(fv, name, ReactivityVBT.State.Passive);
  END IsPassive;

PROCEDURE IsDormant(fv: T; name: Text.T): BOOLEAN RAISES {Error} =
  BEGIN
    RETURN TestReactivity(fv, name, ReactivityVBT.State.Dormant);
  END IsDormant;

PROCEDURE IsVanished(fv: T; name: Text.T): BOOLEAN RAISES {Error} =
  BEGIN
    RETURN TestReactivity(fv, name, ReactivityVBT.State.Vanish);
  END IsVanished;

PROCEDURE TestReactivity (fv: T; name: Text.T; state: ReactivityVBT.State):
  BOOLEAN RAISES {Error} =
  BEGIN
    RETURN state = ReactivityVBT.Get (FindReactivityVBT (fv, name));
  END TestReactivity;

PROCEDURE FindReactivityVBT (fv: T; name: Text.T): FVFilter
  RAISES {Error} =
  VAR v := GetVBT (fv, name);
  BEGIN
    WHILE v # NIL DO
      TYPECASE v OF FVFilter => RETURN v ELSE END;
      v := VBT.Parent (v);
    END;
    RAISE Error ("Cannot find FVFilter");
  END FindReactivityVBT;

PROCEDURE GetBoolean (fv: T; name: TEXT): BOOLEAN
  RAISES {Error, Unimplemented} =
  BEGIN
    TYPECASE GetVBT (fv, name) OF
    | FVBoolean (b) => RETURN BooleanVBT.Get (b)
    | FVChoice (c) => RETURN ChoiceVBT.Get (c) = c
    ELSE
      RAISE Unimplemented
    END
  END GetBoolean;

PROCEDURE GetChoice (fv: T; radioName: TEXT): TEXT
  RAISES {Error, Unimplemented} =
  BEGIN
    TYPECASE GetVBT (fv, radioName) OF
    | FVRadio (r) =>
        TYPECASE Radio.Get (r.radio) OF
        | NULL => RETURN NIL
        | FVChoice (c) => RETURN ItemName (c)
        ELSE
          RAISE Unimplemented
        END
    ELSE
      RAISE Unimplemented
    END
  END GetChoice;

PROCEDURE ItemName (c: FVChoice): TEXT RAISES {Error} =
  VAR cn: ChoiceName := VBT.GetProp (c, TYPECODE (ChoiceName));
  BEGIN
    IF cn = NIL THEN RAISE Error ("Choice button has no name.") END;
    RETURN cn.name
  END ItemName;

PROCEDURE MakeSelected (fv: T; choiceName: TEXT) RAISES {Error} =
  BEGIN
    TYPECASE GetVBT (fv, choiceName) OF
    | FVChoice (c) => ChoiceVBT.Put (c); RETURN
    ELSE
      RAISE Error ("No Choice named " & choiceName)
    END
  END MakeSelected;

PROCEDURE IsSelected (fv: T; choiceName: TEXT): BOOLEAN RAISES {Error} =
  BEGIN
    TYPECASE GetVBT (fv, choiceName) OF
    | FVChoice (c) => RETURN ChoiceVBT.Get (c) = c
    ELSE
      RAISE Error ("No Choice named " & choiceName)
    END
  END IsSelected;

PROCEDURE WhichRadio (fv: T; choiceName: TEXT): TEXT RAISES {Error} =
  BEGIN
    TYPECASE GetVBT (fv, choiceName) OF
    | FVChoice (c) => RETURN ItemName (ChoiceVBT.Get (c))
    ELSE
      RAISE Error ("No Choice named " & choiceName)
    END
  END WhichRadio;

(************************* Generic interactors **********************)

PROCEDURE PutGeneric (fv: T; genericName: TEXT; vbt: VBT.T)
  RAISES {Error} =
  BEGIN
    TYPECASE GetVBT (fv, genericName) OF
    | FVGeneric (v) =>
        IF vbt = NIL THEN
          EVAL Filter.Replace (v, NIL);
          FlexVBT.Set (v, EMPTYSHAPE)
        ELSE
          EVAL Filter.Replace (v, vbt);
          FlexVBT.Set (v, FlexShape.DefaultShape)
        END;
        RETURN
    ELSE
      RAISE Error ("No Generic named " & genericName)
    END
  END PutGeneric;
    
PROCEDURE GetGeneric (fv: T; genericName: TEXT): VBT.T RAISES {Error} =
  BEGIN
    TYPECASE GetVBT (fv, genericName) OF
    | FVGeneric (v) => RETURN Filter.Child (v)
    ELSE
      RAISE Error ("No Generic named " & genericName)
    END
  END GetGeneric;

(************************ Debugging tools ************************)

PROCEDURE NamedVBTs (t: T): List.T =
  BEGIN
    RETURN List.Sort (t.getVBT.toAssocList (), CompareCar)
  END NamedVBTs;
  
PROCEDURE CompareCar ( <* UNUSED *>arg: REFANY; item1, item2: REFANY):
  [-1 .. 1] =
  BEGIN
    RETURN Text.Compare (
             NARROW (item1, List.T).first, NARROW (item2, List.T).first)
  END CompareCar;

<*UNUSED *> (* except during debugging! *)
PROCEDURE DumpTable (fv: T) =
  VAR
    value      : REFANY;
    key        : TEXT;
    alist, pair: List.T;
  BEGIN
    alist := NamedVBTs (fv);
    WHILE alist # NIL DO
      pair := List.Pop (alist);
      key := List.Pop (pair);
      value := pair.first;
      SmallIO.PutText (stderr, key);
      SmallIO.PutText (stderr, " -> ");
      SmallIO.PutText (stderr, RTutils.TypeName (value));
      SmallIO.PutChar (stderr, '\n')
    END
  END DumpTable;

VAR (* CONST *)  AttachmentTypecodes: ARRAY [0 .. 2]OF CARDINAL;

PROCEDURE GetAttachments (fv: T): List.T =
  VAR
    alist     : List.T := NIL;
    ignoreText: TEXT;
    ignoreRef : REFANY;
  PROCEDURE fetch ( <* UNUSED *>data: REFANY; key: TEXT; VAR value: REFANY):
    BOOLEAN =
    VAR property: REFANY;
    BEGIN
      FOR i := FIRST (AttachmentTypecodes) TO LAST (AttachmentTypecodes) DO
        property := VBT.GetProp (value, AttachmentTypecodes [i]);
        IF property # NIL THEN
          List.Push (alist, List.List2 (key, property));
          EXIT
        END
      END;
      RETURN FALSE
    END fetch;
  BEGIN
    EVAL fv.getVBT.enumerate (fetch, NIL, ignoreText, ignoreRef);
    RETURN alist
  END GetAttachments;

PROCEDURE SetAttachments (fv: T; alist: List.T) RAISES {Error} =
  VAR
    name      : TEXT;
    attachment: REFANY;
    pair      : List.T;
  BEGIN
    WHILE alist # NIL DO
      pair := List.Pop (alist);
      name := pair.first;
      attachment := pair.tail.first;
      TYPECASE attachment OF
      | FilterClosureRef (c) => AttachKeyFilter (fv, name, c.filter)
      | FocusClosureRef (c) => AttachFocusAlert (fv, name, c.alert)
      | ClosureRef (c) => Attach (fv, name, c.cl)
      ELSE
        RAISE Error ("Internal error: bad attachment for " & name)
      END
    END
  END SetAttachments;

PROCEDURE InitRuntime () =
  <* FATAL ColorName.NotFound *>
  BEGIN
    AttachmentTypecodes :=
      ARRAY [0 .. 2] OF
        CARDINAL {TYPECODE (FilterClosureRef), TYPECODE (FocusClosureRef),
                  TYPECODE (ClosureRef)};
    MakeEventMiscCodeType := VBT.GetMiscCodeType ("FVRuntime.MakeEvent");
    MakeEventSelection := VBT.GetSelection ("FVRuntime.MakeEvent");
    FocusMiscCodeType := VBT.GetMiscCodeType ("FVRuntime.Focus");
    FocusSelection := VBT.GetSelection ("FVRuntime.Focus");
    cleanState.fgOp :=
      PaintOpCache.FromRGB (
        cleanState.fgRGB, PaintOp.Mode.Accurate, bw := PaintOp.BW.UseFg);
    cleanState.bgOp :=
      PaintOpCache.FromRGB (
        cleanState.bgRGB, PaintOp.Mode.Accurate, bw := PaintOp.BW.UseBg);
    cleanState.lightOp :=
      PaintOpCache.FromRGB (
        cleanState.lightRGB, PaintOp.Mode.Accurate, bw := PaintOp.BW.UseFg);
    cleanState.darkOp :=
      PaintOpCache.FromRGB (
        cleanState.darkRGB, PaintOp.Mode.Accurate, bw := PaintOp.BW.UseFg);
        
    cleanState.fontMetrics := DefaultFontMetrics;
    cleanState.fontName := MetricsToName (cleanState.fontMetrics);
    cleanState.font := Font.FromName (ARRAY OF TEXT {cleanState.fontName});
    
    cleanState.labelFontMetrics := DefaultLabelFontMetrics;
    cleanState.labelFontName := MetricsToName (cleanState.labelFontMetrics);
    cleanState.labelFont :=
      Font.FromName (ARRAY OF TEXT {cleanState.labelFontName});
      
    cleanState.shadow :=
      Shadow.New (cleanState.shadowSz, cleanState.bgOp, cleanState.fgOp,
                  cleanState.lightOp, cleanState.darkOp);
    (* Initial state.menuBar and state.zsplit are set in Init. *)
    
  END InitRuntime;

BEGIN InitParser (); InitRuntime () END FVRuntime.
