INTERFACE RandomGen;

(***************************************************************************)
(*                      Copyright (C) Olivetti 1989                        *)
(*                          All Rights reserved                            *)
(*                                                                         *)
(* Use and copy of this software and preparation of derivative works based *)
(* upon this software are permitted to any person, provided this same      *)
(* copyright notice and the following Olivetti warranty disclaimer are     *) 
(* included in any copy of the software or any modification thereof or     *)
(* derivative work therefrom made by any person.                           *)
(*                                                                         *)
(* This software is made available AS IS and Olivetti disclaims all        *)
(* warranties with respect to this software, whether expressed or implied  *)
(* under any law, including all implied warranties of merchantibility and  *)
(* fitness for any purpose. In no event shall Olivetti be liable for any   *)
(* damages whatsoever resulting from loss of use, data or profits or       *)
(* otherwise arising out of or in connection with the use or performance   *)
(* of this software.                                                       *)
(***************************************************************************)

(* Random number generators *) 

TYPE
  Generator = OBJECT
  METHODS
    int(): INTEGER RAISES {};
    seed(seed: INTEGER) RAISES {};
  END;

(* The 'Generator' type defines two basic operations of a random number
generator.
1) The 'int' method returns a random integer.
2) The 'seed' method is useful in generators which use a pseudo random
sequence; calling the 'seed' argument sets the generator to be at some point in
the sequence. For other random number generators 'seed' may well do nothing.
  Perhaps the most common use of 'seed' is to try and make pseudo random
sequences a little more random by starting them off at a random point.
Typically some genuinely random but expensive to compute value (such as the
system time) is used as a seed so that different runs of a program will not
always cover the same part of the pseudo random sequence.

  This module provides some random number generators plus some procedures which
use simple random number generators to implement slightly more complex
functions, such as returning a random number in a given range.

  Users who implement their own random number generators should note that they
are responsible for making their generator suitable for use with multi threaded
programs (e.g. protect any state by a mutex). The generators provided by this
module are all suitable for use in multi threaded programs. *)


(* Random number generators *)

<*INLINE*> PROCEDURE BBC(): Generator RAISES {};
(* Uses the BBC micro random number generator algorithm *)

<*INLINE*> PROCEDURE ANSI_C(): Generator RAISES {};
(* Uses the ANSI C random number generator algorithm *)


(* The default generator *)

PROCEDURE DefaultGenerator(): Generator RAISES {};
(* return the default generator (currently it is initially 'BBC') *)

PROCEDURE SetDefaultGenerator(g: Generator): Generator RAISES {};
(* sets the default generator to be 'g' and returns the old default *)


(* Procedures which operate on random number generators *)
 
<*INLINE*> PROCEDURE Uniform(
    g: Generator;
    lwb, upb: INTEGER)
    : INTEGER
    RAISES {};
(* return a random integer between the given bounds. The arguments 'lwb' and
'upb' needn't be in the correct order e.g. 'Uniform(7, 3)' is treated as
'Uniform(3, 7)'; both will return a number in the range 3..7 inclusive *)

<*INLINE*> PROCEDURE ZeroTo(g: Generator; limit: CARDINAL): CARDINAL RAISES {};
(* return a random number in the range 0..limit inclusive *)

TYPE
  NonZeroCARDINAL = [1..LAST(CARDINAL)];

<*INLINE*> PROCEDURE OneTo(
    g: Generator;
    limit: NonZeroCARDINAL)
    : NonZeroCARDINAL
    RAISES {};
(* return a random number in the range 1..limit inclusive *)

END RandomGen.
