INTERFACE IOBuffer_priv;

(***************************************************************************)
(*                      Copyright (C) Olivetti 1989                        *)
(*                          All Rights reserved                            *)
(*                                                                         *)
(* Use and copy of this software and preparation of derivative works based *)
(* upon this software are permitted to any person, provided this same      *)
(* copyright notice and the following Olivetti warranty disclaimer are     *) 
(* included in any copy of the software or any modification thereof or     *)
(* derivative work therefrom made by any person.                           *)
(*                                                                         *)
(* This software is made available AS IS and Olivetti disclaims all        *)
(* warranties with respect to this software, whether expressed or implied  *)
(* under any law, including all implied warranties of merchantibility and  *)
(* fitness for any purpose. In no event shall Olivetti be liable for any   *)
(* damages whatsoever resulting from loss of use, data or profits or       *)
(* otherwise arising out of or in connection with the use or performance   *)
(* of this software.                                                       *)
(***************************************************************************)

(* Provides access to the stream buffer of an active stream. For experts only!
Intended to allow fast unsafe put and get routines to be built on streams.

All the procedures in this interface give a checked runtime error if their
stream argument is NIL.

The term 'valid stream' is used to describe a stream that is not uninitialized,
errant, disabled or closed.

The procedures in this interface allow the user to disable and then enable a
stream.

To be disabled a stream must be valid (note that this means it is impossible to
disable a stream that is already disabled). When the stream is disabled none of
the procedures in the IO interface can be used on the stream, apart from a few
enquiry functions - 'Name', 'Properties', 'WhyErrant', 'Closed',
'DescribeError'. Any attempt to use other procedures from IO with a disabled
stream will cause an 'Error' to be raised, and will enable the stream again
(maintaining the invariant that a disabled stream must be valid).

To be enabled a stream must be disabled. Note that this means that immediately
after the stream is enabled it is guaranteed to be valid *)

FROM IO IMPORT Stream, Error;

TYPE
  Buffer = REF ARRAY OF CHAR;

PROCEDURE StreamBuffer(s: Stream): Buffer RAISES {Error};
(* Returns the stream buffer; 's' must be a valid stream *)

PROCEDURE Disable(s: Stream) RAISES {Error};
(* Disables 's'. If an error occurs 'Error' is raised before the stream is
disabled. *)

<*INLINE*> PROCEDURE Enable(s: Stream) RAISES {};
(* Enables 's'. It is a checked runtime error if 's' is not disabled *)

PROCEDURE SinglePut(s: Stream; ch: CHAR) RAISES {Error};
(* A special put procedure which can be used when 's' is disabled; it enables
's', puts the character and then disables 's' again. It is a checked runtime
error if 's' is not disabled *)


TYPE
  Bounds = RECORD low, high: CARDINAL END;

PROCEDURE DisableForGetting(
    s: Stream;
    makeSpace: BOOLEAN)
    : Bounds
    RAISES {Error};
(* Disable 's' and return the bounds of the buffer area available for getting
i.e. the area containing unread characters. If 'makeSpace' is FALSE the bounds
returned may be equal - indicating no unread characters in the buffer. If
'makeSpace' is TRUE and there are no unread characters in the buffer the
buffer is refilled. If 'makeSpace' is TRUE the bounds returned are only equal
if the end of stream is reached.
  If an error occurs 'Error' is raised before the stream is disabled *)

PROCEDURE EnableAfterGet(s: Stream; pos: CARDINAL) RAISES {};
(* Enables 's' and sets its current buffer position to 'pos'. This procedure is
for use after a 'DisableForGetting' call - the user has presumably read
characters from the buffer and now wishes to let the stream know how many have
been read.
  It is a checked runtime error if 's' is not disabled or if 'pos' is not in
the unread area of the buffer *)

PROCEDURE DisableForPutting(
    s: Stream;
    makeSpace: BOOLEAN)
    : Bounds
    RAISES {Error};
(* Disable 's' and return the bounds of the buffer area available for putting
If 'makeSpace' is FALSE the bounds returned may be equal - indicating no space
left in the buffer. If 'makeSpace' is TRUE and there is no space in the buffer
the buffer is flushed. If 'makeSpace' is TRUE the bounds returned are never
equal.
  If is a checked runtime error if 's' is unbuffered or line buffered (use
'SinglePut' to put to such a disabled stream).
  If an error occurs 'Error' is raised before the stream is disabled *)

PROCEDURE EnableAfterPut(s: Stream; pos: CARDINAL) RAISES {};
(* Enables 's' and sets its current buffer position to 'pos'. This procedure is
for use after a 'DisableForPutting' call - the user has presumably put some
characters to the buffer and now wishes to let the stream know how many have
been put.
  It is a checked runtime error if 's' is not disabled or if 'pos' is not in
the area of the buffer available for putting *)

END IOBuffer_priv.
