(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Last modified on Thu Apr 23 18:57:36 PDT 1992 by muller                   *)

UNSAFE MODULE Main;


IMPORT Thread, RTHeapRep, RTHeap;
IMPORT Text, RTHeapPolicy, RTMisc;
IMPORT RTHeapEvent, RTHeapComm, Stdio, Wr, Rd, Fmt;

IMPORT VBT, TextVBT, ButtonVBT, RigidVBT, HVSplit;
IMPORT Trestle, Region, PaintOp, Axis, Split;
IMPORT Rect, Point, ColorName, RGB;

FROM RTHeapRep IMPORT  Space, Type, Promotion, Page, Desc;

<*FATAL ANY*>

(*------------------------------------------------------------ heap state ---*)

TYPE

CONST
  switch = ARRAY Space OF Space { Space.Unallocated, Space.Free,
                                   Space.Current, Space.Free };
      (* after a gc, the space of each page is transformed through this
         array *)

VAR
  collections: INTEGER := 0;

  firstPage: Page := 1;
  lastPage: Page := 0;

  desc := NEW (UNTRACED REF ARRAY OF Desc, 0);


TYPE
  Counter = {unallocated, free, ambiguous, large, frozen, allocated};
VAR
  count := ARRAY Counter OF CARDINAL {0, ..};

PROCEDURE CounterOf (d: Desc): Counter =
  BEGIN
    CASE d.space OF
    | Space.Free => 
        RETURN Counter.free;
    | Space.Current, Space.New =>
        CASE d.promoted OF
        | Promotion.Allocated =>
            RETURN Counter.allocated;
        | Promotion.AmbiguousRoot =>
            RETURN Counter.ambiguous;
        | Promotion.Large =>
            RETURN Counter.large;
        | Promotion.ContainsFrozenRef,
          Promotion.ContainsAccessibleFrozenRef =>
            RETURN Counter.frozen; END;
    | Space.Unallocated => 
        RETURN Counter.allocated; END;
  END CounterOf;

PROCEDURE ComputeAndDisplayCounters () =
  BEGIN
    FOR i := FIRST (count) TO LAST (count) DO
      count [i] := 0; END;
    FOR i := FIRST (desc^) TO LAST (desc^) DO 
      INC (count [CounterOf (desc[i])]); END;
    DisplayCounters ();
  END ComputeAndDisplayCounters;

(*---------------------------------------------------------------- colors ---*)

VAR
  tints: ARRAY Space, Promotion OF PaintOp.T;
  mapBackGround := ComputeColor ("LightLightGray");
  red := ComputeColor ("Red");
  black := ComputeColor ("Black");
  white := ComputeColor ("White");

  gcOnQuad := PaintOp.MakeColorQuad (black, red);
  gcOffQuad := PaintOp.MakeColorQuad (white, black);

PROCEDURE ComputeColor (name: Text.T): PaintOp.T =
  VAR t: RGB.T;
  BEGIN
    t := ColorName.ToRGB (name);
    RETURN PaintOp.FromRGB (t[0], t[1], t[2]);
  END ComputeColor;

PROCEDURE InitColors () =
  BEGIN
    (* initialize the colors ...*)
    tints [Space.Unallocated] :=
        ARRAY Promotion OF PaintOp.T {
          ComputeColor ("Black"), ..};
  

    tints [Space.Current] :=
        ARRAY Promotion OF PaintOp.T {
          ComputeColor ("Green"),
          ComputeColor ("Magenta"),
          ComputeColor ("Violet"),
          ComputeColor ("Violet"),
          ComputeColor ("Magenta")};

    tints [Space.New] :=
        ARRAY Promotion OF PaintOp.T {
          ComputeColor ("LightLightGreen"),
          ComputeColor ("LightLightMagenta"),
          ComputeColor ("LightLightBlue"),
          ComputeColor ("LightLightBlue"),
          ComputeColor ("LightLightMagenta")};

    tints [Space.Free] :=
        ARRAY Promotion OF PaintOp.T {
          ComputeColor ("White"), ..};

  END InitColors; 

(*---------------------------------------------------------------------------*)
  
PROCEDURE DisplayCounters () =
  BEGIN
    LOCK VBT.mu DO
      TextVBT.Put (unallocatedCount, Fmt.Int (count [Counter.unallocated]));
      TextVBT.Put (ambiguousCount, Fmt.Int (count [Counter.ambiguous]));
      TextVBT.Put (frozenCount, Fmt.Int (count [Counter.frozen]));
      TextVBT.Put (allocatedCount, Fmt.Int (count [Counter.allocated]));
      TextVBT.Put (largeCount, Fmt.Int (count [Counter.large]));
      TextVBT.Put (freeCount, Fmt.Int (count [Counter.free]));

      WITH n = count [Counter.ambiguous] 
                 + count [Counter.frozen] + count [Counter.allocated]
                 + count [Counter.large] DO
        TextVBT.Put (active, Fmt.Int (n));
        TextVBT.Put (alloc,  Fmt.Int (n + count [Counter.free])); END; END;
  END DisplayCounters;
  
PROCEDURE UpdateDisplay () = 
  BEGIN
    DisplayCounters ();
    LOCK VBT.mu DO
      TextVBT.Put (policyMinRecovery, Fmt.Int (RTHeapPolicy.minRecovery));
      TextVBT.Put (policyRecoveryRatio,
                   Fmt.Int (TRUNC (RTHeapPolicy.recoveryRatio * 100.0)));
      TextVBT.Put (policyMinIncrement, Fmt.Int (RTHeapPolicy.minIncrement));
      TextVBT.Put (policyGrowthRate, 
                   Fmt.Int (TRUNC (RTHeapPolicy.growthRate * 100.0))); END;

  END UpdateDisplay;


(*------------------------------------------------------------   Heap map ---*)

TYPE 
  HeapMapVBT = VBT.Leaf OBJECT
                 rect := Rect.T {0, 1, 0, 1};
                 side: INTEGER;
                 nCols, nRows: INTEGER := 1;
                 firstSquare: Rect.T;
                 displayedTints: REF ARRAY OF PaintOp.T := NIL;
               OVERRIDES
                 repaint := RepaintHeapMap;
                 reshape := ReshapeHeapMap;
                 shape   := ShapeHeapMap; END;

PROCEDURE LayoutHeapMap (self: HeapMapVBT) =
  (* Given the rectangle to be occupied by the heap map and the number of
     pages to display, compute the size of each square *)

  VAR tryLarger := TRUE; 
      p := MAX (lastPage - firstPage + 1, 1);
      width, height: INTEGER;

  BEGIN
    (* Recompute the layout of the map *)
    width := self.rect.east - self.rect.west;
    height := self.rect.south - self.rect.north; 
    self.side := 1;
    self.nCols := width;
    self.nRows := height;

    WHILE tryLarger DO
      WITH largerSide = self.side + 1,
           largerCols = width DIV largerSide,
           largerRows = height DIV largerSide DO
     
        IF p <= largerCols * largerRows THEN (* ok *)
          self.side := largerSide;
          self.nCols := largerCols;
          self.nRows := largerRows;
        ELSE
          tryLarger := FALSE; END; END; END;
            
    self.firstSquare := Rect.FromCorner (
                          Point.MoveHV (Rect.NorthWest (self.rect),
                                   (width - self.side * self.nCols) DIV 2, 
                                   (height - self.side * self.nRows) DIV 2),
                          self.side, self.side);

  END LayoutHeapMap;
  
PROCEDURE RepaintHeapMap (self: HeapMapVBT; 
			  <*UNUSED*> READONLY rgn: Region.T) = 
  VAR p := 0;
      nbPages := lastPage - firstPage + 1;
      square := self.firstSquare;
  BEGIN
    (* Fill the map with the background color *)
    VBT.PaintTint (self, self.rect, mapBackGround);

    (* redisplay each page *)
    FOR y := 0 TO self.nRows-1 DO
      FOR x := 0 TO self.nCols-1 DO
        IF p < nbPages THEN
          VAR d := desc [p];  sq := square; BEGIN
            INC (sq.north, 1);
            DEC (sq.south, 1);
            IF d.type = Type.Header THEN
              INC (sq.west, 1);
              IF p = nbPages - 1
                   OR desc [p+1].type # Type.Continued 
                   OR desc [p+1].space = Space.Unallocated 
                   OR desc [p+1].space = Space.Free THEN
                DEC (sq.east, 1); END;
            ELSIF p = nbPages - 1
                    OR desc [p+1].type # Type.Continued 
                    OR desc [p+1].space = Space.Unallocated 
                    OR desc [p+1].space = Space.Free THEN
              DEC (sq.east, 1); END;
            VBT.PaintTint (self, square, white);
            VBT.PaintTint (self, sq, 
                           tints [d.space, d.promoted]); END; END;
        INC (p);
        INC (square.east, self.side); 
        INC (square.west, self.side); END;
      square.east := self.firstSquare.east;
      square.west := self.firstSquare.west;
      INC (square.north, self.side);
      INC (square.south, self.side); END;
  END RepaintHeapMap;

PROCEDURE RepaintOnePage (self: HeapMapVBT; page: Page) =
  VAR p := page - firstPage;
      nbPages := lastPage - firstPage + 1;
      row := p DIV self.nCols;
      col := p - row * self.nCols;
      d := desc [p];
      tint := tints [d.space, d.promoted];
      square, sq: Rect.T;
  BEGIN
    square.west := self.firstSquare.west + col * self.side;
    square.east := square.west + self.side;
    square.north := self.firstSquare.north + row * self.side;
    square.south := square.north + self.side;

    sq := square;

    INC (sq.north, 1);
    DEC (sq.south, 1);
    IF d.type = Type.Header THEN
      INC (sq.west, 1);
      IF p = nbPages - 1
          OR desc [p+1].type # Type.Continued 
          OR desc [p+1].space = Space.Unallocated 
          OR desc [p+1].space = Space.Free THEN
        DEC (sq.east, 1); END;
    ELSIF p = nbPages - 1
            OR desc [p+1].type # Type.Continued 
            OR desc [p+1].space = Space.Unallocated 
            OR desc [p+1].space = Space.Free THEN
      DEC (sq.east, 1); END;
    VBT.PaintTint (self, square, white);
    VBT.PaintTint (self, sq, tint);
  END RepaintOnePage;

PROCEDURE ReshapeHeapMap (self: HeapMapVBT; READONLY cd: VBT.ReshapeRec) = 
  BEGIN
    self.rect := cd.new;
    LayoutHeapMap (self);
    RepaintHeapMap (self, Region.T {r := cd.new});
  END ReshapeHeapMap;

PROCEDURE ShapeHeapMap (<*UNUSED*> self: HeapMapVBT; 
                        ax: Axis.T;
                        <*UNUSED*> n: CARDINAL): VBT.SizeRange =
  BEGIN
    IF ax = Axis.T.Hor THEN
      RETURN (VBT.SizeRange {lo := 200, pref := 300, hi := 100*1000});
    ELSE
      RETURN (VBT.SizeRange {lo := 200, pref := 200, hi := 100*1000}); END;
  END ShapeHeapMap;

(*---------------------------------------------------------- various VBTs ---*)

TYPE 
  PatchVBT = VBT.Leaf OBJECT
               color: PaintOp.T;
             OVERRIDES
               repaint := RepaintPatch; END;

PROCEDURE RepaintPatch (self: PatchVBT; READONLY rgn: Region.T) = 
  BEGIN
    VBT.PaintTint (self, rgn.r, self.color);
  END RepaintPatch;

PROCEDURE NewPatchVBT (color: PaintOp.T): VBT.T =
  BEGIN
    RETURN RigidVBT.New (NEW (PatchVBT, color := color),
                         RigidVBT.Shape { 
                            RigidVBT.SizeRange {lo := 5.0, pref := 5.0, 
                                                hi := 5.0},
                            RigidVBT.SizeRange {lo := 0.0,  pref := 2.0,
                                                  hi := 1.0e6}});
  END NewPatchVBT;


PROCEDURE ColorLegendVBT (name: Text.T; 
                          c: PaintOp.T; value: VBT.T): VBT.T =
  BEGIN
    RETURN HVSplit.Cons (Axis.T.Hor,
             NewPatchVBT (c), TextVBT.New (name, 0.0), value);
  END ColorLegendVBT;


PROCEDURE ShowValueVBT (name: Text.T; value: VBT.T): VBT.T =
  BEGIN
    RETURN HVSplit.Cons (Axis.T.Hor,
                         TextVBT.New (name, 0.0),
                         value);
  END ShowValueVBT;


PROCEDURE ControlValueVBT (name: Text.T; valu: VBT.T; 
                           less, more: ButtonVBT.Proc; 
                           right: Text.T := NIL): VBT.T =
  VAR res: VBT.T;
  BEGIN
    res := HVSplit.Cons (Axis.T.Hor,
                         ButtonVBT.New (TextVBT.New ("-"), less, valu),
                         TextVBT.New (name, 0.0),
                         valu);
    IF right # NIL THEN
      Split.AddChild (res, TextVBT.New (right)); END;
    Split.AddChild (res, ButtonVBT.New (TextVBT.New ("+"), more, valu));
    RETURN res;
  END ControlValueVBT;

TYPE
  A = REF RECORD p: PROCEDURE (); END;

PROCEDURE ActionVBT (name: Text.T; action: PROCEDURE ()): VBT.T = 
  BEGIN
    RETURN ButtonVBT.New (TextVBT.New (name), 
                          DoActionVBT, NEW (A, p := action));
  END ActionVBT;

PROCEDURE DoActionVBT (self: ButtonVBT.T;
                           <*UNUSED*> READONLY cd: VBT.MouseRec) =
  BEGIN
    NARROW (VBT.GetProp (self, TYPECODE (A)), A).p ();
  END DoActionVBT;

(*--------------------------------------------------------- Redraw Button ---*)


PROCEDURE GrowHeap () =
  <*UNUSED*> VAR x := NEW (REF ARRAY OF CHAR, RTHeapRep.BytesPerPage * 500);
  BEGIN
  END GrowHeap;

(*------------------------------------------------------- Number Displays ---*)

VAR
  gcs    := TextVBT.New ("");
  active := TextVBT.New ("");
  alloc  := TextVBT.New ("");
  prohibited := TextVBT.New ("");
  
  policyMinRecovery    := TextVBT.New ("");
  policyRecoveryRatio  := TextVBT.New ("");
  policyMinIncrement   := TextVBT.New ("");
  policyGrowthRate     := TextVBT.New ("");

  unallocatedCount := TextVBT.New ("", 0.5, 0.0);
  freeCount      := TextVBT.New ("", 0.5, 0.0);
  ambiguousCount := TextVBT.New ("", 0.5, 0.0);
  frozenCount    := TextVBT.New ("", 0.5, 0.0);
  allocatedCount := TextVBT.New ("", 0.5, 0.0);
  largeCount     := TextVBT.New ("", 0.5, 0.0);
  
(*-------------------------------------------------------------- controls ---*)

VAR 
  root,
    control: VBT.T;
    map    : HeapMapVBT;



PROCEDURE lessMinRecovery (<*UNUSED*>self: ButtonVBT.T;
                           <*UNUSED*> READONLY cd: VBT.MouseRec) =
  BEGIN
    lessConst (RTHeapPolicy.minRecovery);
  END lessMinRecovery;

PROCEDURE moreMinRecovery (<*UNUSED*>self: ButtonVBT.T;
                           <*UNUSED*> READONLY cd: VBT.MouseRec) =
  BEGIN
    moreConst (RTHeapPolicy.minRecovery);
  END moreMinRecovery;

PROCEDURE lessMinIncrement (<*UNUSED*>self: ButtonVBT.T;
                           <*UNUSED*> READONLY cd: VBT.MouseRec) =
  BEGIN
    lessConst (RTHeapPolicy.minIncrement);
  END lessMinIncrement;

PROCEDURE moreMinIncrement (<*UNUSED*>self: ButtonVBT.T;
                           <*UNUSED*> READONLY cd: VBT.MouseRec) =
  BEGIN
    moreConst (RTHeapPolicy.minIncrement);
  END moreMinIncrement;

PROCEDURE lessConst (VAR x: INTEGER) =
  BEGIN
    x := x DIV 2; 
  END lessConst;

PROCEDURE moreConst (VAR x: INTEGER) =
  BEGIN
    IF x = 0 THEN 
      x := 1;
    ELSE 
      x := x * 2; END;
  END moreConst;


PROCEDURE lessRecoveryRatio (<*UNUSED*>self: ButtonVBT.T;
                           <*UNUSED*> READONLY cd: VBT.MouseRec) =
  BEGIN
    RTHeapPolicy.recoveryRatio := 
      MAX (RTHeapPolicy.recoveryRatio - 0.05, 0.0);
  END lessRecoveryRatio;

PROCEDURE moreRecoveryRatio (<*UNUSED*>self: ButtonVBT.T;
                           <*UNUSED*> READONLY cd: VBT.MouseRec) =
  BEGIN
    RTHeapPolicy.recoveryRatio := 
      MIN (RTHeapPolicy.recoveryRatio + 0.05, 1.0);
  END moreRecoveryRatio;

PROCEDURE lessGrowthRate (<*UNUSED*>self: ButtonVBT.T;
                           <*UNUSED*> READONLY cd: VBT.MouseRec) =
  BEGIN
    RTHeapPolicy.growthRate :=  RTHeapPolicy.growthRate / 2.0;
  END lessGrowthRate;

PROCEDURE moreGrowthRate (<*UNUSED*>self: ButtonVBT.T;
                           <*UNUSED*> READONLY cd: VBT.MouseRec) =
  BEGIN
   RTHeapPolicy.growthRate := 2.0 * RTHeapPolicy.growthRate;
  END moreGrowthRate;


VAR
  trestleThread: Thread.T;


PROCEDURE StartAction () =
  BEGIN
    Wr.PutChar (Stdio.stdout, 'g');
    Wr.Flush (Stdio.stdout);
  END StartAction;

PROCEDURE QuitAction () =
  BEGIN
    Trestle.Delete (root);
    RTMisc.Exit (0);
  END QuitAction;


PROCEDURE SetupVBT () =
  BEGIN
    control := HVSplit.New (Axis.T.Ver);
    Split.AddChild (control,
      ColorLegendVBT ("unallocated",
           tints [Space.Unallocated, Promotion.Allocated],
           unallocatedCount),
      ColorLegendVBT ("free",
           tints [Space.Free, Promotion.AmbiguousRoot],
           freeCount),
      ColorLegendVBT ("ambiguous",
           tints [Space.Current, Promotion.AmbiguousRoot],
           ambiguousCount),
      ColorLegendVBT ("frozen",
           tints [Space.Current, Promotion.ContainsFrozenRef],
           frozenCount),
      ColorLegendVBT ("allocated",
           tints [Space.Current, Promotion.Allocated],
           allocatedCount));

    Split.AddChild (control,
      ShowValueVBT ("active = ", active),
      ShowValueVBT ("alloc = ", alloc),
      ShowValueVBT ("gcs = ", gcs),
      ShowValueVBT ("prohibited = ", prohibited));

(*
    Split.AddChild (control,
      ControlValueVBT ("min recovery", policyMinRecovery,
                       lessMinRecovery, moreMinRecovery, "pages"),
      ControlValueVBT ("recovery ratio", policyRecoveryRatio,
                       lessRecoveryRatio, moreRecoveryRatio, "%"),
      ControlValueVBT ("min increment", policyMinIncrement,
                       lessMinIncrement, moreMinIncrement, "pages"), 
      ControlValueVBT ("growth rate", policyGrowthRate,
                       lessGrowthRate, moreGrowthRate, "%"),
      RigidVBT.New (TextVBT.New (""),
        RigidVBT.Shape { 
          RigidVBT.SizeRange {lo := 50.0, pref := 50.0, hi := 50.0},
            RigidVBT.SizeRange {lo := 0.0, pref := 0.0, hi := 1.0e6}}));
*)

    Split.AddChild (control,
(*      ActionVBT ("collect now", RTHeapRep.CollectNow),
      ActionVBT ("enable collection", RTHeap.EnableCollection),
      ActionVBT ("disable collection", RTHeap.DisableCollection),
      ActionVBT ("grow", GrowHeap),
      ActionVBT ("redraw", UpdateDisplay), *)
      ActionVBT ("start", StartAction),
      ActionVBT ("quit", QuitAction));

    map     := NEW (HeapMapVBT);
    root    := HVSplit.Cons (Axis.T.Hor, control, map);

    Trestle.Install (root);
  
    trestleThread := Thread.Fork (NEW (Thread.SizedClosure, 
                                  stackSize := 100000,
                                  apply := AwaitDelete));
  END SetupVBT;

PROCEDURE AwaitDelete (<*UNUSED*> self: Thread.Closure): REFANY RAISES {} = 
  BEGIN
    Trestle.AwaitDelete (root);
    RETURN NIL;
  END AwaitDelete;

(*---------------------------------------------------------------------------*)

BEGIN
  InitColors ();
  SetupVBT ();

  TRY
    LOOP
      VAR e := RTHeapComm.Receive (Stdio.stdin); BEGIN

        CASE e.kind OF
        | RTHeapEvent.Kind.Bye =>
            EXIT;

        | RTHeapEvent.Kind.CollectionStart =>
            INC (collections);
            TextVBT.SetFont (gcs, TextVBT.GetFont (gcs), gcOnQuad);
            TextVBT.Put (gcs, Fmt.Int(collections));

        | RTHeapEvent.Kind.CollectionProhibited =>
            TextVBT.Put (prohibited, Fmt.Int (e.nb));

        | RTHeapEvent.Kind.PromotedRoots =>
             (* don't care *)

        | RTHeapEvent.Kind.Flip =>
            FOR i := FIRST (desc^) TO LAST (desc^) DO 
              desc [i].space := switch [desc [i].space]; END;
            RepaintHeapMap (map, Region.T {r := map.rect});

        | RTHeapEvent.Kind.CollectionStop  =>
            TextVBT.SetFont (gcs, TextVBT.GetFont (gcs), gcOffQuad);
            ComputeAndDisplayCounters ();

        | RTHeapEvent.Kind.Grow =>
            VAR newFirstPage, newLastPage: Page;
                newDesc: UNTRACED REF ARRAY OF Desc; BEGIN

              IF firstPage = 1 AND lastPage = 0 THEN
                newFirstPage := e.first;
                newLastPage := e.first + e.nb - 1;
              ELSE
                newFirstPage := MIN (e.first, firstPage);
                newLastPage := MAX (e.first + e.nb - 1, lastPage); END;

              newDesc := NEW (UNTRACED REF ARRAY OF Desc,
                              newLastPage - newFirstPage + 1);

              FOR p := e.first TO e.first + e.nb - 1 DO
                newDesc [p - newFirstPage].space := Space.Free; END;
              
              IF NOT (firstPage = 1 AND lastPage = 0) THEN
                SUBARRAY (newDesc^, firstPage - newFirstPage,
                          lastPage - firstPage + 1) := desc^;
                
                FOR p := e.first + e.nb TO firstPage - 1 DO
                  newDesc [p - newFirstPage].space := Space.Unallocated; END;
                
                FOR p := lastPage + 1 TO e.first - 1 DO
                  newDesc [p - newFirstPage].space := Space.Unallocated; END; END;
                
              desc := newDesc;
              firstPage := newFirstPage;
              lastPage := newLastPage; END;

            LayoutHeapMap (map);
            RepaintHeapMap (map, Region.T {r := map.rect});
            ComputeAndDisplayCounters ();

        | RTHeapEvent.Kind.ChangePages =>
            desc [e.first - firstPage] := e.desc;
            e.desc.type := RTHeapRep.Type.Continued; 
            FOR p := e.first + 1 TO e.first + e.nb - 1 DO 
              desc [p - firstPage] := e.desc; END;
            FOR p := e.first TO e.first + e.nb - 1 DO
              RepaintOnePage (map, p);  END; END; END; END;
          
  EXCEPT 
  | Rd.EndOfFile => END;

  EVAL Thread.Join (trestleThread);
END Main.

