(* Copyright (C) 1992, Digital Equipment Corporation *)
(* All rights reserved. *)
(* See the file COPYRIGHT for a full description. *)
(* *)
(* by Steve Glassman *)
(* Last modified on Mon Nov  2 12:45:44 PST 1992 by steveg *)

<*PRAGMA LL*>

(* An "Image" is a screen-independent resource that displays as
   a colored pixmap. *)

INTERFACE Image;

IMPORT
  Pixmap, Rect, ScrnColorMap, ScrnPixmap, TrestleComm, VBT;

TYPE
  T = Pixmap.T;

PROCEDURE Unscaled (raw: Raw): T;
(* Returns a "T" which will display as "raw".  "raw.bits" will not
   be scaled regardless of the screen's resolution. *)

PROCEDURE Scaled (raws: ARRAY OF Raw; tolerance := 0.25; maxScale := 4): T;
(* Returns a "T" which will scale and display "raw[i].bits".  "i" will be
   chosen so that the raw pixmap will have the resolution closest to an
   integer multiple within "tolerance" of the screen's resolution.  The
   pixmap will then be scaled by the appropriate integer multiple and
   displayed.  No pixmap will scaled by a factor greater than "maxScale".*)

VAR (* CONST *)
  BitmapColors: Colors; 
    (* = Colors{ScrnColorMap.RGB{0.0, 0.0, 0.0},
                ScrnColorMap.RGB{1.0, 1.0, 1.0}}; *)
  (* convenience for converting Pixmap.Raw to Image.Raw:
     rawImage := NEW(Image.Raw, bits := rawPixmap, colors := Image.BitmapColors)
  *)

PROCEDURE FromScrnPixmap (spm: ScrnPixmap.T; st: VBT.ScreenType): Raw
  RAISES {TrestleComm.Failure};
(* Returns a screen-independent "Raw" which describes the pixmap "spm" when
   displayed on "st". *)

TYPE
  Colors = REF ARRAY OF
                 RECORD
                   pixel: CARDINAL;
                   rgb  : ScrnColorMap.RGB
                 END;
  Raw = OBJECT
          bits      : Pixmap.Raw;
          colors    : Colors;
          colorMode                := ScrnColorMap.Mode.Normal;
          xres, yres: REAL         := 75.0; (* in pixels per inch *)
        END;
(* The "colors" array gives the rgb value for each pixel value represented
   in the pixmap "bits".

   The colors are matched using "colorMode" as described in ScrnColorMap.

   "xres" and "yres" give the nominal resolution of the pixmap.  Depending
   on the registration procedure used, the "bits" may be scaled to better
   match the resolution of the screen. *)

PROCEDURE NewRaw (dpth: INTEGER; READONLY bnds: Rect.T; cntColors: INTEGER):
  Raw;
(* Allocate and return a raw image with the given "depth" and "bounds" and
   a color array of the size "cntColors". *)

END Image.
