(* Copyright (C) 1992, Digital Equipment Corporation *)
(* All rights reserved. *)
(* See the file COPYRIGHT for a full description. *)
(* *)
(* by Steve Glassman *)
(* Last modified on Mon Nov  2 13:09:34 PST 1992 by steveg *)

<*PRAGMA LL*>

(* An "Image" is a screen-independent resource that displays as
   a colored pixmap. *)

MODULE Image;

IMPORT Axis, Palette, Pixmap, Point, Rect, ScreenType, ScrnColorMap, ScrnPixmap,
       TrestleComm, VBT;

TYPE UnscaledClosure = Palette.PixmapClosure OBJECT
    raw: Raw;
  OVERRIDES
    apply := ApplyUnscaled
  END;

TYPE ScaledClosure = Palette.PixmapClosure OBJECT
    raws: REF ARRAY OF Raw;
    tolerance: REAL;
    maxScale: INTEGER;
  OVERRIDES
    apply := ApplyScaled
  END;

PROCEDURE Unscaled (raw: Raw): T =
  BEGIN
    RETURN Palette.FromPixmapClosure(NEW(UnscaledClosure, raw := raw));
  END Unscaled;

PROCEDURE Scaled (raws: ARRAY OF Raw; tolerance := 0.25; maxScale := 4):
  T =
  VAR new := NEW(REF ARRAY OF Raw, NUMBER(raws));
  BEGIN
    new^ := raws;
    RETURN Palette.FromPixmapClosure(
             NEW(ScaledClosure, raws := new, tolerance := tolerance,
                 maxScale := maxScale));
  END Scaled;

TYPE PixelMap = ARRAY [0 .. 255] OF INTEGER;

<* FATAL ScrnColorMap.Failure *>
PROCEDURE ApplyScaled (cl: ScaledClosure; st: ScreenType.T): ScrnPixmap.T =
  VAR
    xres                    := st.res[Axis.T.Hor] * 25.4;
    yres                    := st.res[Axis.T.Hor] * 25.4;
    closest                 := -1;
    xyScaleClosest          := cl.maxScale;
    xyRemClosest  : INTEGER;
    tol                     := cl.tolerance;
    t             : REAL;
  BEGIN
    (* find the raw closest to an integer multiple of the screen
       resolution *)
    LOOP
      FOR i := 0 TO LAST(cl.raws^) DO
        t := MAX(0.0, 1.0 - tol);
        WITH xs    = ROUND(xres * t / cl.raws[i].xres),
             ys    = ROUND(yres * t / cl.raws[i].yres),
             xyMax = MAX(xs, ys),
             xyRem = TRUNC(MAX(ABS(FLOAT(xs) * cl.raws[i].xres - xres),
                         ABS(FLOAT(ys) * cl.raws[i].yres - yres))) DO
          IF (xyMax < xyScaleClosest)
               OR (xyMax = xyScaleClosest AND xyRem < xyRemClosest) THEN
            closest := i;
            xyScaleClosest := xyMax;
            xyRemClosest := xyRem;
          END;
        END;
      END;
      IF closest # -1 THEN EXIT END;
      tol := 2.0 * tol;
    END;
    RETURN (ScaleRaw(
              st, cl.raws[closest],
              MIN(cl.maxScale, ROUND(xres * t / cl.raws[closest].xres)),
              MIN(cl.maxScale, ROUND(yres * t / cl.raws[closest].yres))));
  END ApplyScaled;

<* FATAL ScrnColorMap.Failure *>
PROCEDURE ApplyUnscaled (cl: UnscaledClosure; st: ScreenType.T):
  ScrnPixmap.T =
  BEGIN
    RETURN (ScaleRaw(st, cl.raw, 1, 1));
  END ApplyUnscaled;

TYPE
  ScaleAction = {UseMap, UseCmapOn24Bit, BW, CvtRGBTo24Bit, UsePixels};

PROCEDURE CvtRGBTo24Bit (rgb: ScrnColorMap.RGB): INTEGER =
  BEGIN
    RETURN ROUND(rgb.r * 255.0) * 256 * 256 + ROUND(rgb.g * 255.0) * 256
             + ROUND(rgb.b * 255.0)
  END CvtRGBTo24Bit;

PROCEDURE Cvt24BitToBW (pix: INTEGER): INTEGER =
  BEGIN
    IF pix MOD 256 >= 128 THEN
      RETURN 1
    ELSE
      pix := pix DIV 256;
      IF pix MOD 256 >= 128 THEN
        RETURN 1
      ELSE
        pix := pix DIV 256;
        IF pix MOD 256 >= 128 THEN RETURN 1 END
      END
    END;
    RETURN 0
  END Cvt24BitToBW;

PROCEDURE Cvt24BitToRGB (pix: INTEGER): ScrnColorMap.RGB =
  VAR r, g, b: REAL;
  BEGIN
    b := FLOAT(pix MOD 256) / 255.0;
    pix := pix DIV 256;
    g := FLOAT(pix MOD 256) / 255.0;
    pix := pix DIV 256;
    r := FLOAT(pix MOD 256) / 255.0;
    RETURN ScrnColorMap.RGB{r, g, b}
  END Cvt24BitToRGB;

PROCEDURE ScaleRaw (st: ScreenType.T; raw: Raw; xMul, yMul: INTEGER):
  ScrnPixmap.T =
  VAR
    map : PixelMap;
    cmap: ScrnColorMap.T;
    pix : INTEGER;
    dest: Point.T;
    a                    := ScaleAction.UsePixels;
  BEGIN
    TRY
      WITH oldB = raw.bits.bounds,
           newB = Rect.T{
                    north := oldB.north, west := oldB.west, south :=
                    oldB.north + yMul * (oldB.south - oldB.north), east :=
                    oldB.west + xMul * (oldB.east - oldB.west)},
           new = ScrnPixmap.NewRaw(raw.bits.depth, newB) DO
        IF raw.bits.depth <= 8 THEN
          IF st.depth <= 8 THEN
            a := ScaleAction.UseMap;
            IF raw.bits.depth = 1 THEN
              map[0] := 0;
              map[1] := 1;
            ELSIF raw.bits.depth <= 8 THEN
              cmap := st.cmap.standard();
              FOR i := 0 TO LAST(raw.colors^) DO
                TRY
                  map[raw.colors[i].pixel] :=
                    cmap.fromRGB(raw.colors[i].rgb, raw.colorMode);
                EXCEPT
                | ScrnColorMap.Failure =>
                    map[raw.colors[i].pixel] := 1; (* ? *)
                END;
              END;
            END;
          ELSIF st.depth = 24 THEN
            a := ScaleAction.CvtRGBTo24Bit
          END;
        ELSIF raw.bits.depth = 24 THEN
          IF cmap = NIL THEN
            a := ScaleAction.BW
          ELSE
            a := ScaleAction.UseCmapOn24Bit
          END
        END;
        FOR i := 0 TO xMul - 1 DO
          FOR h := raw.bits.bounds.west TO raw.bits.bounds.east - 1 DO
            FOR j := 0 TO yMul - 1 DO
              FOR v := raw.bits.bounds.north
                  TO raw.bits.bounds.south - 1 DO
                pix := raw.bits.get(Point.T{h, v});
                dest := Point.T{xMul * h + i, yMul * v + j};
                CASE a OF
                | ScaleAction.UseMap => pix := map[pix];
                | ScaleAction.UseCmapOn24Bit =>
                    TRY
                      pix :=
                        cmap.fromRGB(Cvt24BitToRGB(pix), raw.colorMode);
                    EXCEPT
                    | ScrnColorMap.Failure => pix := 1; (* ? *)
                    END;
                | ScaleAction.BW => pix := Cvt24BitToBW(pix);
                | ScaleAction.CvtRGBTo24Bit =>
                    pix := CvtRGBTo24Bit(raw.colors[pix].rgb);
                | ScaleAction.UsePixels =>
                END;
                new.set(dest, pix);
              END;
            END;
          END;
        END;
        RETURN st.pixmap.load(new)
      END;
    EXCEPT
    | TrestleComm.Failure => RETURN Palette.ResolvePixmap(st, Pixmap.Solid)
    END
  END ScaleRaw;

EXCEPTION BadDepth;

PROCEDURE FromScrnPixmap (pm: ScrnPixmap.T; st: VBT.ScreenType): Raw
  RAISES {TrestleComm.Failure} =
  VAR
    raw                                          := NEW(Raw);
    map                                          := PixelMap{-1, ..};
    cntPix                                       := 0;
    colors: ARRAY [0 .. 255] OF ScrnColorMap.RGB;
    entry : ARRAY [0 .. 1] OF ScrnColorMap.Entry;
    cmap                                         := st.cmap.standard();
    spm                                          := pm.localize(pm.bounds);
  BEGIN
    IF spm.depth <= 8 THEN
      raw.bits := ScrnPixmap.NewRaw(spm.depth, spm.bounds);
      FOR h := spm.bounds.west TO spm.bounds.east - 1 DO
        FOR v := spm.bounds.north TO spm.bounds.south - 1 DO
          WITH pix = spm.get(Point.T{h, v}) DO
            IF map[pix] = -1 THEN
              map[pix] := cntPix;
              entry[0].pix := pix;
              cmap.read(entry);
              colors[cntPix] := entry[0].rgb;
              INC(cntPix);
            END;
            raw.bits.set(Point.T{h, v}, map[pix]);
          END;
        END;
      END;
      raw.colors := NEW(Colors, cntPix);
      FOR i := 0 TO cntPix - 1 DO
        raw.colors[i].pixel := i;
        raw.colors[i].rgb := colors[i];
      END;
    ELSIF spm.depth = 24 THEN
      raw.bits := ScrnPixmap.NewRaw(spm.depth, spm.bounds);
      FOR h := spm.bounds.west TO spm.bounds.east - 1 DO
        FOR v := spm.bounds.north TO spm.bounds.south - 1 DO
          raw.bits.set(Point.T{h, v}, spm.get(Point.T{h, v}));
        END;
      END;
    ELSE
      RAISE BadDepth;
    END;
    RETURN raw;
  END FromScrnPixmap;

PROCEDURE NewRaw (dpth: INTEGER; READONLY bnds: Rect.T; cntColors: INTEGER):
  Raw =
  BEGIN
    RETURN NEW(Raw, bits := ScrnPixmap.NewRaw(dpth, bnds),
               colors := NEW(Colors, cntColors));
  END NewRaw;

BEGIN
  BitmapColors := NEW(Colors, 2);
  BitmapColors[0].pixel := 0;
  BitmapColors[0].rgb := ScrnColorMap.RGB{0.0, 0.0, 0.0};
  BitmapColors[1].pixel := 1;
  BitmapColors[1].rgb := ScrnColorMap.RGB{1.0, 1.0, 1.0};
END Image.
