(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)
(*                                                                           *)
(* Last modified on Fri Jul 31 23:11:42 PDT 1992 by meehan *)
(*      modified on Tue Jun 16 13:08:20 PDT 1992 by muller *)
(*      modified on Sat Jun 13 08:21:29 1992 by mhb *)

(* The "Shadow" interface contains the basic definitions for VBT
   classes that implement a Motif-like, 3-D look.

   A {\em shadow} is like a border, but it gives the visual
   illusion of ``raising'' a rectangle above the background,
   ``lowering'' a rectangle into the background, drawing a
   ``ridge'' above the background, or chisling a ``groove'' into
   the background.

   These visual effects are actually quite simple to accomplish
   by drawing parts of the border using a dark variant of the
   background color, and by drawing other parts using a light
   variant of the background color.

   For example, to give the impression that an object is raised
   above its background, the north and west borders re drawn
   using a light color, whereas the south and east border are
   drawn in a dark color.  To draw a ``ridge'', the north and
   west shadows start out in the light color, and, halfway,
   switch to the dark color.  Analogously, the south and east
   shadows start out dark and switch to a light color.

   For maximum effectiveness, the child's background should be a
   color whose saturation level is about 50\%, and the light and
   dark shadows should colors with the same hue and lightness,
   but with saturation levels of 25\% and 75\% respectively.

   The following chart summarizes how the visual effects are 
   accomplished:

             \begin{center}
             \begin{tabular}{l|l|l}

             {\em Style} & {\em North/West} & {\em South/East}\\
             \hline
                 Flat      &   Background  &  Background\\
                 Raised    &   Light       &  Dark\\
                 Lowered   &   Dark        &  Light\\
                 Ridged    &   Light/Dark  &  Dark/Light\\
                 Chiseled  &   Dark/Light  &  Light/Dark

             \end{tabular}
             \end{center}

     See Figure~\ref{fig:shadows}.
   *)

INTERFACE Shadow;

IMPORT PaintOp, VBT;

TYPE
  Points = REAL;
  T = PaintOp.ColorScheme OBJECT
        size : Points;
        light: PaintOp.T;
        dark : PaintOp.T;
      END;
(* Shadows will occupy "ABS(size)" points.  A negative size is
   interpreted by some "VBT" classes as a signal to give feedback
   in a 2-D style.  For instance, when they are activated,
   "SwitchVBT"s display an inverted image, rather than a recessed
   shadow.*)

TYPE Style = {Flat, Raised, Lowered, Ridged, Chiseled};

PROCEDURE New (size : Points := 2.0;
               bg            := PaintOp.Bg;
               fg            := PaintOp.Fg;
               light         := PaintOp.Fg;
               dark          := PaintOp.Fg  ): T;
(* Return a newly allocated "Shadow.T" with the specified fields.
   All of the paint ops should be tints, arranged so that on a
   monochrome screen "bg" draws in background, while "fg",
   "light", and "dark" draw in foreground. *)

PROCEDURE SetSize  (shadow: T; newSize: Points): T;
PROCEDURE SetBg    (shadow: T; newBg:    PaintOp.T): T;
PROCEDURE SetFg    (shadow: T; newFg:    PaintOp.T): T;
PROCEDURE SetLight (shadow: T; newLight: PaintOp.T): T;
PROCEDURE SetDark  (shadow: T; newDark:  PaintOp.T): T;
(* Return a newly allocated copy of "shadow", with the
   specified field replaced with a new value. *)

PROCEDURE Supported (shadow: T; v: VBT.T): BOOLEAN;
(* Return whether "shadow" will appear 3-dimensional on 
   "v".  Two conditions must hold: "v" must be on a color screen and
   "shadow.size" must be positive. *)

VAR (* CONST *) None: T;
(* This variable is really a constant for
| New(0.0, PaintOp.Bg, PaintOp.Fg, PaintOp.Fg, PaintOp.Fg)
   Because "None" is not a constant, it cannot be the
   default value of a procedure argument.  Therefore, we adopt the
   following convention: when a parameter whose type is
   "Shadow.T" has a default value of "NIL", the procedure will
   use "Shadow.None" instead. *)

END Shadow.

