/* -*-C-*-
********************************************************************************
*
* File:         x2jet.c
* RCS:          $Header: $
* Description:  xpr support for HP LaserJet and PaintJet printers
* Author:       Larry Rupp, HP Graphics Technology Division
* Created:      Fri Jul 15 15:22:26 1988
* Modified:     Fri Aug 26 15:18:55 1988 (Larry Rupp) ler@hpfcler
* Language:     C
* Package:      N/A
* Status:       Released to MIT
*
* (c) Copyright 1988, Hewlett-Packard Company.
*
********************************************************************************
*/
/********************************************************

Copyright (c) 1988 by Hewlett-Packard Company
Copyright (c) 1988 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this software 
and its documentation for any purpose and without fee is hereby 
granted, provided that the above copyright notice appear in all 
copies and that both that copyright notice and this permission 
notice appear in supporting documentation, and that the names of 
Hewlett-Packard or  M.I.T.  not be used in advertising or publicity 
pertaining to distribution of the software without specific, written 
prior permission.

********************************************************/

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/XWDFile.h>

#include "xpr.h"

#ifndef	TRUE
#  define	FALSE	0
#  define	TRUE	1
#endif

/* default printable page area (inches) */
#define STDWIDTH 8.0
#define STDHEIGHT 10.5

/* header & trailer character cell size (centipoints) */
#define CHARWIDTH 720
#define CHARHEIGHT 1200

#define XWDHEADERSIZE	(sizeof(XWDFileHeader))
#define XCOLORSIZE  	(sizeof(XColor))


typedef struct { long width, height; } Area;
typedef struct { long x, y; } Location;

char *malloc();


static Area limit;	/* image clip limits (dots) */
static Area page;	/* printable page size (centipoints) */

static Location headerloc;	/* centipoint location of header string */
static Location trailerloc;	/* centipoint location of trailer string */
static Location imageloc;	/* centipoint location of image */

static int headerlimit;		/* number of chars which will printed for */
static int trailerlimit;	/*  the image's header/trailer strings    */

static XWDFileHeader xwd_header;

static XColor *xwd_colors;

static char *xwd_image;



void fatal_err (s)
char * s;
{
  fprintf(stderr, s);
  exit(1);
}


/* Computes the centipoint width of one printer dot. */
#define dot_centipoints(s,d)	((7200.0 * s) / d)



void set_image_limits (scale, density, orient, print_area)
int scale, density;
enum orientation orient;
Area print_area;
{
  Area print_dots;
  double dotsize;

  /* Set dotsize to the centipoint width of one printer dot. */
  dotsize = dot_centipoints(scale, density);

  if (orient == PORTRAIT) {
    print_dots.width = print_area.width / dotsize;
    print_dots.height = print_area.height / dotsize;
  } else {
    print_dots.height = print_area.width / dotsize;
    print_dots.width = print_area.height / dotsize;
  }

  limit.width = (print_dots.width < xwd_header.pixmap_width)
		? print_dots.width : xwd_header.pixmap_width;
  limit.height = (print_dots.height < xwd_header.pixmap_height)
		? print_dots.height : xwd_header.pixmap_height;

  if ((limit.width != xwd_header.pixmap_width)
      || (limit.height != xwd_header.pixmap_height))
    fprintf(stderr,"xpr: Warning: %ld x %ld image clipped to %ld x %ld.\n",
	    xwd_header.pixmap_width, xwd_header.pixmap_height,
	    limit.width, limit.height);
}



void set_header_trailer_limits (header, trailer, printwidth)
char *header, *trailer;
long printwidth;
{
  /* Determine the number of header and trailer characters
   * that will fit into the available printing area.
   */
  headerlimit = header ? (((strlen(header) * CHARWIDTH) <= printwidth)
			  ? strlen(header) : (printwidth / CHARWIDTH))
		       : 0;
  if (header && headerlimit != strlen(header))
    fprintf(stderr,"xpr: Warning: Header string clipped to %d characters.\n");

  trailerlimit = trailer ? (((strlen(trailer) * CHARWIDTH) <= printwidth)
			    ? strlen(trailer) : (printwidth / CHARWIDTH))
			 : 0;
  if (trailer && trailerlimit != strlen(trailer))
    fprintf(stderr,"xpr: Warning: Trailer string clipped to %d characters.\n");
}


void set_print_locations (scale, density, top, left,
			  header, trailer, orient, position_on_page)
int scale, density;
int top, left;
char *header, *trailer;
enum orientation orient;
int position_on_page;
{
  Area image;
  double dotsize;

  /* Set dotsize to the centipoint width of one printer dot. */
  dotsize = dot_centipoints(scale, density);

  /* Compute the centipoint size of the clipped image area. */
  if (orient == PORTRAIT) {
    image.width = limit.width * dotsize;
    image.height = limit.height * dotsize;
  } else {
    image.height = limit.width * dotsize;
    image.width = limit.height * dotsize;
  }

  if (position_on_page) {
    /* set vertical positions */
    imageloc.y = (top >= 0)
	      ? top * 24 + ((header) ? CHARHEIGHT : 0)
	      : ((page.height - ((header) ? CHARHEIGHT : 0)
		  - image.height - ((trailer) ? CHARHEIGHT : 0)) / 2)
		+ ((header) ? CHARHEIGHT : 0);
    headerloc.y = imageloc.y - CHARHEIGHT / 4;
    trailerloc.y = imageloc.y + image.height + (3 * CHARHEIGHT) / 4;

    /* set horizontal positions */
    if (left >= 0)
      headerloc.x = imageloc.x = trailerloc.x = left * 24;
    else {
      headerloc.x = (page.width - headerlimit * CHARWIDTH) / 2;
      imageloc.x = (page.width - image.width) / 2;
      trailerloc.x = (page.width - trailerlimit * CHARWIDTH) / 2;
    }
  }
}


int scale_raster (density, orient, print_area)
int density;
enum orientation orient;
Area print_area;
{
  Area image;
  int h_scale, v_scale;

  /* Set the image dimensions to the number of centipoints that would be
   * required for printing at the selected density.
   */
  if (orient == PORTRAIT) {
    image.width = xwd_header.pixmap_width * 7200 / density;
    image.height = xwd_header.pixmap_height * 7200 / density;
  } else {
    image.height = xwd_header.pixmap_width * 7200 / density;
    image.width = xwd_header.pixmap_height * 7200 / density;
  }

  /* Calculate the maximum image multiplier along
   * the horizontal and vertical dimensions.
   */
  h_scale = print_area.width / image.width;
  v_scale = print_area.height / image.height;

  /* If the image can be expanded, return the lesser of the horizontal and
   * vertical multipliers.  Otherwise, the image will not completely fit
   * the available print area, so just return 1 as the expansion factor.
   */
  return (((h_scale > 0) && (v_scale > 0))
	  ? ((h_scale<v_scale) ? h_scale : v_scale)
	  : 1);
}


scale_and_orient_image (scale, density, width, height, left, top,
			header, trailer,
			orient, position_on_page, paintjet)
int *scale, *density;
int width, height, left, top;  /* in 300ths of an inch */
char *header, *trailer;
enum orientation *orient;
int position_on_page, paintjet;
{
  Area usable;

  /* Determine printable area expressed in centipoints.  There are 7200
   * centipoints to the inch.  The width and height parameters passed in
   * are expressed in 300ths of an inch, therefore a 24x conversion factor
   * is used on the parameter values.  The default page dimensions STDWIDTH
   * and STDHEIGHT are expressed in inches so must be multiplied by 7200
   * to convert to centipoints.
   */
  page.width = (width >= 0) ? width * 24 : STDWIDTH * 7200;
  page.height = (height >= 0) ? height * 24 : STDHEIGHT * 7200;

  /* Determine the area usable for the image.  This area will be smaller
   * than the total printable area if margins or header/trailer strings
   * have been specified.  Margins, like width and height discussed above,
   * are expressed in 300ths of an inch and must be converted to centipoints.
   * Header and trailer strings each reduce the available image height
   * by 1/6 inch, or 1200 centipoints (aka CHARHEIGHT).
   */
  usable.width = page.width - ((left > 0) ? (left * 24) : 0);
  usable.height = page.height - ((top > 0) ? (top * 24) : 0)
		  - ((header) ? CHARHEIGHT : 0)
		  - ((trailer) ? CHARHEIGHT : 0);

  /* Quit here if there is no usable image space. */
  if ((usable.width <= 0) || (usable.height <= 0)) {
    fatal_err("xpr: no space available on page for image.\n");
    exit(1);
  }

  /* Determine image orientation.  The orientation will only be changed if
   * it was not specified by a command line option.  Portrait mode will be
   * used if either the usable printing area or the image area are square.
   * Portrait mode will also be used if the long dimensions of the usable
   * printing area and the image area match, otherwise landscape mode is
   * used.  Portrait mode really means "don't rotate" and landscape mode
   * means "rotate".
   */
  if (*orient == UNSPECIFIED) {
    if ((usable.width == usable.height)
	|| (xwd_header.pixmap_width == xwd_header.pixmap_height))
      *orient = PORTRAIT;
    else
      *orient = ((usable.width < usable.height)
		 == (xwd_header.pixmap_width < xwd_header.pixmap_height))
	? PORTRAIT : LANDSCAPE;
  }

  /* Set the dots-per-inch print density if it was not specified */
  if (*density <= 0)
    *density = paintjet ? 90 : 300;

  /* Fit image to available area if scale was not specified */
  if (*scale <= 0)
    *scale = scale_raster(*density, *orient, usable);

  /* Determine image clipping limits */
  set_image_limits(*scale, *density, *orient, usable);

  /* Determine header/trailer string length clipping */
  set_header_trailer_limits(header, trailer, usable.width);

  /* Calculate locations for page layout */
  set_print_locations(*scale, *density, top, left,
		      header, trailer, *orient, position_on_page);

}


#ifdef XCOLORBUG
#define XFULLINTENSITY (unsigned short) 0xFF00
#else
#define XFULLINTENSITY (unsigned short) 0xFFFF
#endif

#define BLACK 1
#define WHITE 0
#define EMPTY -1
#define MAX_PJ_COLOR 16

#define RGBmatch(x,i)	(((x).red == i) && ((x).green == i) && ((x).blue == i))

long *colormap;		/* This "colormap" array is used to map from the
			 * Xcolor array (xwd_colors) index numbers into a
			 * pjcolor index number.			*/

long pjcolor[MAX_PJ_COLOR];

static int color_warning_given = FALSE;



void swap_black_and_white ()
{
  /* Reverse black and white in the Xcolor structure array. */

  XColor *color;
  int n;

  for (n=xwd_header.ncolors, color=xwd_colors;  n>0;  n--, color++)
    if (RGBmatch(*color, XFULLINTENSITY))
      color->red = color->green = color->blue = 0;
    else if (RGBmatch(*color, 0))
      color->red = color->green = color->blue = XFULLINTENSITY;
}


void reset_color_mapping ()
{
  int n;
  long *cmap;

  for (n=xwd_header.ncolors, cmap=colormap;  n>0;  n--, cmap++)
    *cmap = EMPTY;

  for (n=0; n<MAX_PJ_COLOR; n++)
    pjcolor[n] = EMPTY;
}



#define Intensity(x)	((x).red * 0.30 + (x).green * 0.59 + (x).blue * 0.11)



void prepare_color_mapping (invert, paintjet, cutoff)
int invert, paintjet;
unsigned int cutoff;
{
  int n;
  long *cmap;
  XColor *color;

  if (!(colormap = (long *) malloc(xwd_header.ncolors * sizeof(long))))
    fatal_err("xpr: Could not allocate memory for X-to-printer colormap.\n");

  /* For PaintJet, color map assignment will be done one line at a time.
   * So for now just reset the X-to-printer color mapping arrays and
   * interchange the Xcolor structure's black and white if the -rv command
   * line option was specified.
   */
  if (paintjet) {
    reset_color_mapping();
    if (invert)
      swap_black_and_white();
  } else
  /* For LaserJet, map each color to black or white based upon the
   * combined intensity of the RGB components.  Note that the normal
   * non-reversed (-rv) LaserJet mapping will represent light areas
   * of the screen as black on the paper.
   */
    for (n=xwd_header.ncolors, color=xwd_colors, cmap=colormap;  n>0;
	 n--, color++, cmap++)
      *cmap = (Intensity(*color) < cutoff)
	      ? (invert ? BLACK : WHITE)
	      : (invert ? WHITE : BLACK);
}


/* On a PaintJet printer, the programmable color intensity ranges are:
 *
 *	red:	4..90		green:	4..88		blue:	6..85
 *
 * The following macros map the 0..65535 intensity ranges of X colors
 * into the PaintJet's ranges.
 */

#define fixred(x)	(x / 762 + 4)
#define fixgreen(x)	(x / 780 + 4)
#define fixblue(x)	(x / 829 + 6)

#define is_grey(x)	(((x).red == (x).green) && ((x).red == (x).blue))



void select_grey (level, r, g, b)
int level, *r, *g, *b;
{
  /* Forced selection of a grey.  This is done since the PaintJet does
   * not do very well when picking greys, they tend to become pink!
   */
  if (level > 66) {  /* white */
    *r = 90;    *g = 88;    *b = 85;
  } else if (level > 35) {
    *r = 43;    *g = 43;    *b = 45;
  } else if (level > 21) {
    *r = 25;    *g = 25;    *b = 33;
  } else if (level > 15) {
    *r = 15;    *g = 16;    *b = 18;
  } else if (level > 11) {
    *r = 14;    *g = 14;    *b = 18;
  } else if (level > 6) {
    *r =  6;    *g =  7;    *b =  8;
  } else {  /* black */
    *r =  4;    *g =  4;    *b =  6;
  }
}


int load_printer_color (out, index)
FILE *out;
long index;
{
  int n, red, blue, green;
    
  if (colormap[index] != EMPTY)
    /* printer has already been programmed for this color index */
    return(1);	/* "success" */
  else {
    /* search for an unused PaintJet mapping entry */
    for (n=0; n<MAX_PJ_COLOR; n++) {
      if (pjcolor[n] == EMPTY) {
	red = fixred(xwd_colors[index].red);
	green = fixgreen(xwd_colors[index].green);
	blue = fixblue(xwd_colors[index].blue);
	if (is_grey(xwd_colors[index]))  /* assist grey selection */
	  select_grey(red, &red, &green, &blue);
	/* download color to printer */
	fprintf(out,"\033*v%dA", red);
	fprintf(out,"\033*v%dB", green);
	fprintf(out,"\033*v%dC", blue);
	fprintf(out,"\033*v%dI", n);
	/* record that this is now programmed */
	pjcolor[n] = index;
	colormap[index] = n;
	/* return "success" */
	return(1);
      }
    }
    /* unable to find or program this color, return "failure" */
    return(0);
  }
}


int load_line_colors (out, line, length)
FILE *out;
long *line;
int length;
{
  for (; length>0; length--, line++)
    if (!load_printer_color(out, *line))
      return(0);
  return(1);
}



void download_colors (out, line, length)
FILE *out;
long *line;
int length;
{
  if (!load_line_colors(out, line, length)) {
    reset_color_mapping();
    if (!load_line_colors(out, line, length) && !color_warning_given) {
      fprintf(stderr,"xpr: Warning: Cannot print all image colors.\n");
      color_warning_given = TRUE;
    }
  }
}


void read_xwd_data (in)
FILE *in;
{
#   define WINDOW_NAME_ALLOC	32
    int window_name_size;
    int image_size;
    int n;
    char window_name [WINDOW_NAME_ALLOC];

    /* Read in XWDFileHeader structure */
    if (fread((char*) &xwd_header, 1, XWDHEADERSIZE, in) != XWDHEADERSIZE)
    	fatal_err("xpr: Could not read xwd file's header.\n");

    /* Skip over window name */
    window_name_size = xwd_header.header_size - XWDHEADERSIZE;
    while (window_name_size > 0) {
    	n = window_name_size > WINDOW_NAME_ALLOC
    	    ? WINDOW_NAME_ALLOC : window_name_size;
    	if (fread(window_name, 1, n, in) != n)
    	    fatal_err("xpr: Could not read xwd file's window name.\n");
    	window_name_size -= n;
    }

    /* Allocate space for xwd color structures */
    if (!(xwd_colors = (XColor*) malloc(sizeof(XColor) * xwd_header.ncolors)))
    	fatal_err("xpr: Could not allocate memory for xwdfile color table.\n");

    /* Read in xwd color structures */
    for (n = 0; n < xwd_header.ncolors; n++)
    	if (fread(&xwd_colors[n], 1, XCOLORSIZE, in) != XCOLORSIZE)
    	    fatal_err("xpr: Could not read xwd file's color table.\n");

    /* Allocate space for xwd image */
    if (xwd_header.pixmap_format == ZPixmap)
    	image_size = 1;
    else if (xwd_header.pixmap_format == XYPixmap)
    	image_size = xwd_header.pixmap_depth;
    else
    	fatal_err("xpr: Image in xwd file is not in Z or XY pixmap format.\n");
    image_size *= xwd_header.bytes_per_line * xwd_header.pixmap_height;
    if (!(xwd_image = malloc(image_size)))
    	fatal_err("xpr: Could not allocate memory for xwd file's image.\n");

    /* Read in xwd image */
    if (fread(xwd_image, 1, image_size, in) != image_size)
    	fatal_err("xpr: Could not read xwd file's image.\n");

}


write_image_prefix (out, scale, density, header, paintjet, position_on_page,
		    initial_formfeed, orient)
FILE *out;
int scale, density;
char *header;
int paintjet, position_on_page, initial_formfeed;
enum orientation orient;
{
  if (initial_formfeed)
    fprintf(out,"\014");

  /* Write out header & positioning commands */
  if (header) {
    if (position_on_page)
      fprintf(out,"\033&a%dH\033&a%dV",
	      /* headerloc x & y are written in decipoints */
	      (int) headerloc.x / 10, (int) headerloc.y / 10);
    fprintf(out,"%s\n", header);
  }

  /* Prepare printer for raster graphics: */

  /* Write image positioning commands */
  if (position_on_page)
    fprintf(out,"\033&a%dH\033&a%dV",
	    /* imageloc x & y are written in decipoints */
	    (int) imageloc.x / 10, (int) imageloc.y / 10);
  
  /* Set printer resolution */
  fprintf(out,"\033*t%dR", density);

  /* Enable all four "planes" for PaintJet */
  if (paintjet)
    fprintf(out,"\033*r4U");

  /* Switch to raster graphics mode */
  fprintf(out,"\033*r1A");

  /* Set picture width for PaintJet */
  if (paintjet)
    fprintf(out,"\033*r%dS",
	    ((int) (orient == PORTRAIT) ? limit.width : limit.height)
	    * scale);
}


write_image_suffix (out, trailer, position_on_page)
FILE *out;
char *trailer;
int position_on_page;
{
  /* Exit raster graphics mode */
  fprintf(out,"\033*rB");

  /* Write out trailer & positioning commands */
  if (trailer) {
    if (position_on_page)
      fprintf(out,"\033&a%dH\033&a%dV",
	      /* trailerloc x & y are written in decipoints */
	      (int) trailerloc.x / 10, (int) trailerloc.y / 10);
    fprintf(out,"%s\n", trailer);
  }
}


long Z_image_pixel (x, y)
int x, y;
{
  int pixel_bytes, offset;
  unsigned char *image;

  pixel_bytes = xwd_header.bits_per_pixel >> 3;
  offset = (x * pixel_bytes) + (y * xwd_header.bytes_per_line);
  image = (unsigned char *) &xwd_image[offset];

  switch (pixel_bytes) {
  case 1:
    return (*image);
  case 2:
    return ((xwd_header.byte_order == MSBFirst)
	    ? (*image++ << 8 | *image)
	    : (*image++ | *image << 8));
  case 3:
    return ((xwd_header.byte_order == MSBFirst)
	    ? (*image++ << 16 | *image++ << 8 | *image)
	    : (*image++ | *image++ << 8 | *image << 16));
  case 4:
    return ((xwd_header.byte_order == MSBFirst)
	    ? (*image++ << 24 | *image++ << 16 | *image++ << 8 | *image)
	    : (*image++ | *image++ << 8 | *image++ << 16 | *image << 24));
  }
}


long XY_image_pixel (x, y)
int x, y;
{
  int plane_start, line_start, bytes_per_bitmap_unit, bitmap_unit_start,
      byte_within_bitmap_unit, offset, byte_mask;

  plane_start = (xwd_header.pixmap_depth - 1) * xwd_header.pixmap_height
		* xwd_header.bytes_per_line;
  line_start = xwd_header.bytes_per_line * y;
  bytes_per_bitmap_unit = xwd_header.bitmap_unit >> 3;
  bitmap_unit_start = (x / xwd_header.bitmap_unit) * bytes_per_bitmap_unit;
  byte_within_bitmap_unit = (xwd_header.byte_order == MSBFirst)
    ? (x % xwd_header.bitmap_unit) >> 3
    :  bytes_per_bitmap_unit - ((x % xwd_header.bitmap_unit) >> 3) - 1;

  offset = plane_start + line_start + bitmap_unit_start
	   + byte_within_bitmap_unit;

  byte_mask = (xwd_header.bitmap_bit_order == MSBFirst)
      ? 0x80 >> (x % 8) : 0x01 << (x % 8);

  return(xwd_image[offset] & byte_mask ? 1 : 0);
}


void write_raster_line (out, scale, paintjet, line, length)
FILE *out;
int scale;
int paintjet;
long *line;
int length;
{
  int planes = (paintjet) ? 4 : 1;
  int raster_bytes = (length * scale + 7) / 8;
  register int bytebits, n, p, e, bit;
  long *lp;
  char *line_colors, *raster_line;
  register char *lc, *rl, byte;
  
  if (!(line_colors = malloc(length))
      || !(raster_line = malloc(raster_bytes * planes)))
    fatal_err("xpr: Could not allocate raster line memory.\n");

  if (paintjet)
    download_colors(out, line, length);

  /* Map the line's colors into output color index numbers */
  for (n=0, lc=line_colors, lp=line;  n<length;  n++)
    *lc++ = (char) colormap[*lp++];

  for (p=0;  p<planes;  p++) {
    bytebits = 0;
    n = length;
    lc = line_colors;
    rl = &raster_line[raster_bytes * p];
    while (n-- > 0) {
      bit = (*lc++ >> p) & 0x01;
      e = scale;
      while (e--) {
	byte = (byte << 1) | bit;
	bytebits++;
	if (bytebits == 8) {
	  *rl++ = byte;
	  bytebits = 0;
	}
      }
    }
    if (bytebits)
      *rl = byte << (8 - bytebits);
  }

  e = scale;
  while (e--) {
    for (p=0;  p<planes;  p++) {
      fprintf(out,"\033*b%d%c", raster_bytes, (p+1 == planes) ? 'W' : 'V');
      fwrite(&raster_line[raster_bytes * p], 1, raster_bytes, out);
    }
  }

  free(line_colors);
  free(raster_line);
}


void write_portrait_Z_image (out, scale, paintjet)
FILE *out;
int scale;
int paintjet;
{
  int x, y;
  int width = limit.width;
  int height = limit.height;
  long *line, *lp;

  if (!(line = (long *) malloc(width * sizeof(long))))
    fatal_err("xpr: Could not allocate memory for image line buffer.\n");

  for (y=0;  y<height;  y++) {

    for (x=0, lp=line;  x<width;  x++)
      *lp++ = Z_image_pixel(x,y);

    write_raster_line(out, scale, paintjet, line, width);
  }
}



void write_landscape_Z_image (out, scale, paintjet)
FILE *out;
int scale;
int paintjet;
{
  int x, y;
  int width = limit.height;
  int height = limit.width;
  long *line, *lp;

  if (!(line = (long *) malloc(width * sizeof(long))))
    fatal_err("xpr: Could not allocate memory for image line buffer.\n");

  for (x=0;  x<height;  x++) {

    for (y=width-1, lp=line;  y>=0;  y--)
      *lp++ = Z_image_pixel(x,y);

    write_raster_line(out, scale, paintjet, line, width);
  }
}


void write_portrait_XY_image (out, scale, paintjet)
FILE *out;
int scale;
int paintjet;
{
  int x, y;
  int width = limit.width;
  int height = limit.height;
  long *line, *lp;

  if (!(line = (long *) malloc(width * sizeof(long))))
    fatal_err("xpr: Could not allocate memory for image line buffer.\n");

  for (y=0;  y<height;  y++) {

    for (x=0, lp=line;  x<width;  x++)
      *lp++ = XY_image_pixel(x,y);

    write_raster_line(out, scale, paintjet, line, width);
  }
}



void write_landscape_XY_image (out, scale, paintjet)
FILE *out;
int scale;
int paintjet;
{
  int x, y;
  int width = limit.height;
  int height = limit.width;
  long *line, *lp;

  if (!(line = (long *) malloc(width * sizeof(long))))
    fatal_err("xpr: Could not allocate memory for image line buffer.\n");

  for (x=0;  x<height;  x++) {

    for (y=width-1, lp=line;  y>=0;  y--)
      *lp++ = XY_image_pixel(x,y);

    write_raster_line(out, scale, paintjet, line, width);
  }
}


void write_Z_image (out, scale, orient, paintjet)
FILE *out;
int scale;
enum orientation orient;
int paintjet;
{
  if (orient == PORTRAIT) {
    write_portrait_Z_image(out, scale, paintjet);
  } else
    write_landscape_Z_image(out, scale, paintjet);
}



void write_XY_image (out, scale, orient, paintjet)
FILE *out;
int scale;
enum orientation orient;
int paintjet;
{
  if (xwd_header.ncolors > 2)
    fatal_err("xpr: XY format image, multiplane images must be Z format.\n");

  if (orient == PORTRAIT) {
    write_portrait_XY_image(out, scale, paintjet);
  } else
    write_landscape_XY_image(out, scale, paintjet);
}



void write_image (out, scale, orient, paintjet)
FILE *out;
int scale;
enum orientation orient;
int paintjet;
{
  switch (xwd_header.pixmap_format) {
  case XYPixmap:
    write_XY_image(out, scale, orient, paintjet);  break;
  case  ZPixmap:
    write_Z_image(out, scale, orient, paintjet);   break;
  default: 
    fatal_err("xpr: image not in XY or Z format.\n");
  }
}


void x2jet(in, out, scale, density, width, height, left, top,
	   header, trailer, orient, invert,
	   initial_formfeed, position_on_page,
	   paintjet, cutoff)
FILE *in, *out;
int scale, density;
int width, height, left, top;  /* in 300ths of an inch */
char *header, *trailer;
enum orientation orient;
int invert, initial_formfeed, position_on_page, paintjet;
unsigned int cutoff;
{
  read_xwd_data(in);

  prepare_color_mapping(invert, paintjet, cutoff);

  scale_and_orient_image(&scale, &density, width, height, left, top,
			 header, trailer,
			 &orient, position_on_page, paintjet);

  write_image_prefix(out, scale, density, header, paintjet, position_on_page,
		     initial_formfeed, orient);
  
  write_image(out, scale, orient, paintjet);

  write_image_suffix(out, trailer, position_on_page);

  fclose(out);
}
