/* grMain.c -
 *
 * Copyright 1988 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software (the graphics, textio, and windows modules of the Magic
 * system) and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.  
 *
 * This file contains a few core variables and routines for
 * manipulating color graphics displays.  Its main function is
 * to provide a central dispatch point to various routines for
 * different display types.
 */

#ifndef lint
static char sccsid[] = "@(#)grMain.c	4.16 MAGIC (Berkeley) 12/1/85";
#endif  not lint

/*
 * The following display types are currently suported by Magic:
 *
 *	UCB512		An old AED512 with the Berkeley microcode roms,
 *			with attached bitpad (SummaGraphics Bitpad-One).
 *
 *	UCB512N		A new AED512 with the Berkeley microcode roms,
 *			with attached SummaGraphics Bitpad-one.
 *
 *    	AED767         	An AED767 with a SummaGraphics Bit Pad One.
 *                    	Because of a lack of features in this device,
 *                    	programable cursors and BitBlt do not work.
 *                    	Many thanks to Norm Jouppi and DECWRL for doing
 *                    	this port.
 *
 *	AED1024		An AED1024 with a SummaGraphics Mouse and rev. D roms.
 *			Because of a lack of features in this device,
 *			programable cursors do not work.  Many thanks to 
 *			Peng Ang and LSI Logic Corp for doing this port.
 *
 *	UCB1024		An AED1024 with SummaGraphics Mouse and the special
 *			AED ROMs that provide the same features as the
 *			UCB512 ROMs.
 *
 *	NULL		A null device for running Magic without using
 *			a graphics display.  This device does nothing
 *			when its routines are called.
 *
 *	SUN120		A Sun Microsystems workstation, model Sun2/120 with
 *			a separate colorboard (/dev/cgone0) and the
 *			Sun optical mouse.  Also works on some old Sun1s with
 *			the 'Sun2 brain transplant'.
 *
 *	SUNBW		A black & white Sun, such as a Sun2.
 *			Because this device only has one bit-plane, Magic will
 *			leave little white spots on the screen after erasing
 *			the box or highlight areas.
 *
 *	SUN160		A Sun with one screen -- in color.
 *
 * To port Magic to another type of display, you need to add its name to
 * the table 'grDisplayTypes' and then add a pointer to an initialization
 * routine to 'grInitProcs'.  The initialization routine will fill in all
 * of the graphics routine pointers so that they point to procedures that
 * can handle the new display type.  All calls to device-specific 
 * procedures are made by indirecting through these pointers.
 */

#include <stdio.h>
#include <sgtty.h>
#include <sys/time.h>
#include <errno.h>
#include <ctype.h>
#include "magic.h"
#include "textio.h"
#include "geometry.h"
#include "graphics.h"
#include "windows.h"
#include "graphicsInt.h"

#ifdef	SUN2
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sun/fbio.h>
#endif

extern char *getenv();
extern int errno;

#define FAVORITE_DISPLAY	"NULL"	/* Default display type */

/* The following rectangle is describes the display area and is available
 * to the user of this module.
 */
global Rect GrScreenRect = {0, 0, 0, 0};

/* The first of the following tables defines the legal
 * display types and the second table defines an
 * initialization routine for each type.
 *
 * These entries MUST be all upper case, since what the user types will
 * be converted to upper case before comparison.
 */

static char *grDisplayTypes[] = {
#ifdef SUN2
    "SUN120",
    "SUN160",
    "SUNBW",
#endif
#ifdef X
    "XWIND",
#else
    "UCB512",
    "UCB512N",
    "AED767",
    "AED1024",
    "UCB1024",
#endif
    "NULL",
    NULL};
#ifdef X
extern xSetDisplay();
#else
extern aedSetDisplay();
extern sunSetDisplay();
extern sunWSetDisplay();
#endif
extern nullSetDisplay();

static (*(grInitProcs[]))() = {
#ifdef SUN2
    sunSetDisplay,
    sunWSetDisplay,
    sunWSetDisplay,
#endif
#ifdef X
    xSetDisplay,
#else
    aedSetDisplay,	/* Handles all AEDs (UCB512s, AED767s, and AED1024s) */
    aedSetDisplay,
    aedSetDisplay,
    aedSetDisplay,
    aedSetDisplay,
#endif
    nullSetDisplay,
    NULL};

/* The following variables are pointers to the various graphics
 * procedures.  The macros in graphics.h cause these pointers
 * to be indirected through when calls occur to graphics procedures.
 * This indirection allows for several display types to be supported
 * by a single version of Magic.  The pointers are initially NULL,
 * but are rewritten by the various graphics initializers.
 */

Void (*GrLockPtr)()		= NULL;
Void (*GrUnlockPtr)()		= NULL;
Void (*GrInitPtr)()		= NULL;
Void (*GrClosePtr)()		= NULL;
Void (*GrSetCMapPtr)()		= NULL;

Void (*GrEnableTabletPtr)()	= NULL;
Void (*GrDisableTabletPtr)()	= NULL;
Void (*GrSetCursorPtr)()	= NULL;
Void (*GrTextSizePtr)()		= NULL;
Void (*GrDrawGlyphPtr)()	= NULL;
Void (*GrBitBltPtr)()		= NULL;
int  (*GrReadPixelPtr)()	= NULL;
Void (*GrFlushPtr)()		= NULL;
Void (*GrCreateWindowPtr)()	= NULL;
Void (*GrDeleteWindowPtr)()	= NULL;
Void (*GrDamagedPtr)()	= NULL;

/* variables similar to the above, except that they are only used
 * internal to the graphics package
 */
Void (*grPutTextPtr)()		= NULL;
int (*grGetCharSizePtr)()	= NULL;
Void (*grSetSPatternPtr)()	= NULL;
Void (*grDefineCursorPtr)()	= NULL;
bool (*grDrawGridPtr)()		= NULL;
Void (*grDrawLinePtr)()		= NULL;
Void (*grSetWMandCPtr)()	= NULL;
Void (*grFillRectPtr)()		= NULL;
Void (*grSetStipplePtr)()	= NULL;
Void (*grSetLineStylePtr)()	= NULL;
Void (*grSetCharSizePtr)()	= NULL;

/* The following variables are set by initialization routines for the
 * various displays.  They are strings that indicate what kind of
 * dstyle, cmap and cursor files should be used for this display.  Almost
 * all of the displays are happy with the default values given below.
 * Note:  a NULL grCMapType means that this display doesn't need a
 * color map (it's black-and-white).
 */

char *grDStyleType = "7bit";
char *grCMapType = "7bit";
char *grCursorType = "bw";

/* The following variable is a device-dependent limit on how many
 * stipples are permitted.  It defaults to 16, but may be reset
 * by device initializers to a larger (or smaller) number, depending
 * on what the devices can actually support.  Must not be greater
 * than GR_NUM_STIPPLES.
 */

int grMaxStipples = 16;

/* Procedure to print text on stdout and stderr.  It is located here so that
 * graphics drivers and do something special with it if they want.
 * This routine should only be called by the textio module.
 */
extern Void fprintf();
Void (*GrFprintfPtr)() = fprintf;

/* Procedure to change the default terminal state. It gets passed 2 pointers,
 * one to the terminal input mode structure, and one to the terminal literal
 * characters structure.  See textio module for details.
 */
Void (*GrTermStatePtr)() = NULL;


/* Procedures called just before and after Magic is suspended (via ^Z). */
extern Void grNullProc();
Void (*GrStopPtr)() = grNullProc;
Void (*GrResumePtr)() = grNullProc;


/*---------------------------------------------------------
 * GrSetDisplay:
 *	This routine sets a display type, opens files,  and initializes the
 *	display.
 *
 * Results:
 *	TRUE is returned if the display was found and initialized
 *	successfully.  If the type didn't register, or the file is 
 *	NULL, then FALSE is returned.
 *
 * Side Effects:
 *	Tables are set up to control which display routines are
 *	used when communcating with the display.  The display
 *	is initialized and made ready for action.
 *---------------------------------------------------------
 */

bool
GrSetDisplay(type, outName, mouseName)
char *type;			/* Name of the display type. */
char *outName;			/* Filename used for communciation with 
				 * display. */
char *mouseName;		/* Filename used for communciation 
				 * with tablet. */

{
    char **ptr;
    char *cp;
    int i;

    if (outName == NULL) 
    {
	TxError("No graphics device specified.\n");
	return FALSE;
    }
    if (mouseName == NULL)
    {
	TxError("No mouse specified.\n");
	return FALSE;
    }

    /* Convert display type to upper case. */
    for (cp = type; *cp; cp++) { if (islower(*cp)) *cp = toupper(*cp); }

    /* See if the display type is in our table. */
    ptr = grDisplayTypes;
    for (i=0; ; i++)
    {
	if (*ptr == NULL) return FALSE;
	if (strcmp(*ptr, type) == 0) break;
	ptr++;
    }

    /* Call the initialization procedure. */
    return (*(grInitProcs[i]))(type, outName, mouseName);
}


/*
 * ----------------------------------------------------------------------------
 *
 * grSunFbDepth --
 *
 *	Find the depth of the Sun's frame buffer.
 *
 * Results:
 *	An integer describing the depth of the display.  Returns 0 if there
 *	were any problems (such as not being on a sun).
 *
 * Side Effects:
 *	None.
 *
 * ----------------------------------------------------------------------------
 */

grSunFbDepth(device)
    char *device;		/* The device (screen) to check on. */
{
#ifdef	SUN2
    int fd;
    struct fbtype fbinfo;

    ASSERT(device != NULL, "grSunFbDepth");
    fd = open(device, O_RDWR);
    if (fd < 0)
    {
	TxError("Error opening '%s', error #%d\n", device, errno);
	MainExit(2);
    }
    if (ioctl(fd, FBIOGTYPE, &fbinfo) == -1)
    {
	TxError("Ioctl error on '%s', error #%d\n", device, errno);
	MainExit(3);
    }
    close(fd);

    return fbinfo.fb_depth;
#else
    return 0;
#endif
}


/*
 * ----------------------------------------------------------------------------
 * GrGuessDisplayType:
 *
 *	Try to guess what sort of machine we are on, and set the display
 *	ports and type appropriately.  This info is overrided by
 *	~cad/lib/displays and by command line switches.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Modifies the strings passed in.
 * ----------------------------------------------------------------------------
 */

void
GrGuessDisplayType(graphics, mouse, display, monitor)
    char **graphics;		/* default device for sending out graphics */
    char **mouse;		/* default device for reading mouse (tablet) */
    char **display;		/* default type of device (AED, etc...) */
    char **monitor;		/* default type of monitor (pale, std) */
{
    bool onSun;			/* Are we on a Sun? */
    bool have2sunScreens;	/* do we have a color Sun board? */
    bool haveSuntools;		/* is Suntools running? */

    *graphics = NULL;
    *mouse = NULL;
    *display = NULL;
    *monitor = "std";

    /* Check for signs of suntools. */
    onSun = (access("/dev/win0", 0) == 0);
    haveSuntools = (getenv("WINDOW_PARENT") != NULL);
    have2sunScreens = FALSE;

#ifdef	SUN2
    {
	/* See if an auxiliary Sun color screen is alive and well. */
	int fd;
	fd = open("/dev/cgone0", O_RDWR);
	if (fd > -1)
	{
	    close(fd);
	    have2sunScreens = TRUE;
	}
    }
#endif

    if (onSun && !haveSuntools) {
	/* GUESS: we are on a sun but not running suntools */
	TxError("You are on a Sun but not running suntools.\n");
	*mouse = *graphics = NULL;
	*display = "NULL";
    }
    else if (onSun && have2sunScreens) {
	/* GUESS:  probably a SUN120 with color board */
	*mouse = *graphics = "/dev/cgone0";
	*display = "SUN120";
    } 
    else if (onSun && (grSunFbDepth("/dev/fb") >= 8)) {
	/* GUESS:  probably a SUN160 (main screen is color) */
	*mouse = *graphics = "/dev/fb";
	*display = "SUN160";
	*monitor = "std";
    } 
    else if (onSun) {
	/* CATCH ALL FOR SUNs: probably a black & white sun */
	*mouse = *graphics = "/dev/fb";
	*display = "SUNBW";
    } 
    else {
	/* GUESS:  who knows, maybe a VAX? */
	*mouse = *graphics = NULL;
	*display = FAVORITE_DISPLAY;
    }
}


/*
 * ----------------------------------------------------------------------------
 * grFgets --
 *
 *	Just like fgets, except that it times out after 20 seconds, and prints
 *	a warning message.  After one second a warning message is also 
 *	printed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 * ----------------------------------------------------------------------------
 */

char *
grFgets(str, n, stream, name)
#ifdef X
{
    /* All input for X is handled through the X envent queue */
}
#else
    char *str;
    int n;
    FILE *stream;
    char *name;		/* The user name of the stream, for the error msg */
{
    int fn;
    char *newstr;
    struct timeval threeSec, twentySecs;
    extern int errno;

    threeSec.tv_sec = 3;	
    threeSec.tv_usec = 0;
    twentySecs.tv_sec = 20;	
    twentySecs.tv_usec = 0;

    fn = 1 << fileno(stream);
    newstr = str;
    n--;
    if (n < 0) return (char *) NULL;

    while (n > 0)
    {
	int f;
	char ch;
        int sel;

	f = fn;
	sel = select(20, &f, (int *) NULL, (int *) NULL, &threeSec);
	if  (sel == 0)
	{
	    TxError("The %s is responding slowly, or not at all.\n", name);
	    TxError("I'll wait for 20 seconds and then give up.\n");
	    TxError("On AEDs, typing return on the AED keyboard may help.\n");
	    f = fn;
	    sel = select(20, &f, (int *) NULL, (int *) NULL, &twentySecs);
	    if (sel == 0)
	    {
		TxError("The %s did not respond.\n", name);
		return (char *) NULL;
	    }
	    else if (sel < 0)
	    {
		if (errno == EINTR) {
		    TxError("Timeout aborted.\n");
		}
		else
		{
		    perror("magic");
		    TxError("Error in reading the %s\n", name);
		}
		return (char *) NULL;
	    }
	    else
		TxError("The %s finally responded.\n", name);
	}
	else if (sel < 0)
	{
	    if (errno != EINTR)
	    {
		perror("magic");
		TxError("Error in reading the %s\n", name);
		return (char *) NULL;
	    }
	    /* else try again, back to top of the loop */
	    continue;
	}

	ch = getc(stream);
	*newstr = ch;
	n--;
	newstr++;
	if (ch == '\n')
	    break;
    }

    *newstr = '\0';
    return str;
}
#endif


/*---------------------------------------------------------------------------
 * grNullProc:
 *
 *	A procedure of the type 'Void' that does absolutely nothing.
 *	Used when we need to point a procedure pointer to something, but
 *	don't want it to do anything.
 *
 * Results:	
 *	None.
 *
 * Side Effects:
 *	None.
 *
 *----------------------------------------------------------------------------
 */

Void
grNullProc()
{
}
