/*
 * DVI previewer for X.
 *
 * Eric Cooper, CMU, September 1985.
 *
 * Code derived from dvi-imagen.c.
 *
 * Modified for X.10 by Bob Scheifler, MIT LCS, January 1986.
 *
 */
#ifndef lint
static char *dv_c = "$Header: xdvi.c,v 1.10 88/07/30 19:13:17 eichin Exp $";
#endif 	lint

#include <sys/types.h>
#ifndef X10
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#else X10
#include <X/Xlib.h>
#endif X10
#include <stdio.h>
#include <ctype.h>
#include "dvi.h"
#include "pxl.h"

#ifndef X10
Display *DISP;
int SCRN;
/*
 * Data structure similar to X10 OpaqueFrame...
 */
typedef unsigned long Pixel;
typedef struct _frame {
	short x, y;		/* where to create the window */
	short width, height;	/* width and height */
	short bdrwidth;		/* border width */
	Pixel border;		/* border pixmap */
	Pixel background;	/* background */
} Frame;
#else X10
#define	XDisplayName(d)	"display"
#endif X10

/* These are probably site dependent */
#define FONT_DIRECTORY	"/usr/lib/tex/fonts"
#define FONT_SUFFIX_PXL	".%dpxl"
#define FONT_SUFFIX_PK	".%dpk"
#ifndef FONTPATH
#define FONTPATH	"TEXFONTS"
#endif /* FONTPATH */
#ifndef DEFAULT_FONT_PATH
#define DEFAULT_FONT_PATH	"/usr/sipb/lib/tex/fonts:/usr/lib/tex/fonts"
#endif /* DEFAULT_FONT_PATH */
FILE *fopen_path();
FILE *fopen_font_path();
char *font_path;

#define PAPER_WIDTH	ROUNDUP(17*pixels_per_inch,shrink_factor*2)
#define PAPER_HEIGHT	ROUNDUP(11*pixels_per_inch,shrink_factor)
#define X_PAGE_OFFSET	ROUNDUP(pixels_per_inch,shrink_factor)
#define Y_PAGE_OFFSET	ROUNDUP(pixels_per_inch,shrink_factor)

#define pixel_round(x)      ((long) (conv * (double) (x) + 0.5))
#define dvi_round(x)        ((long) ((double) (x) / conv + 0.5))

#define one(fp)     num (fp, 1)
#define sone(fp)    snum(fp, 1)
#define two(fp)     num (fp, 2)
#define stwo(fp)    snum(fp, 2)
#define four(fp)    num (fp, 4)
#define sfour(fp)   snum(fp, 4)

typedef unsigned char ubyte;

struct frame {
	long pxl_h, dvi_h, pxl_v, dvi_v, w, x, y, z;
};

struct frame *stack;
int stackp;

#define PXL_H   stack[stackp].pxl_h
#define PXL_V   stack[stackp].pxl_v
#define DVI_H   stack[stackp].dvi_h
#define DVI_V   stack[stackp].dvi_v
#define WW      stack[stackp].w
#define XX      stack[stackp].x
#define YY      stack[stackp].y
#define ZZ      stack[stackp].z

#define DBG_BITMAP	0x1
#define DBG_DVI		0x2
#define DBG_PK          0x4
#define DBG_BATCH       0x8
#define DBG_ALL		(DBG_BITMAP|DBG_DVI|DBG_PK)

/*
 * Command line flags.
 */
int debug = 0;
int list_fonts = 0;

int pixels_per_inch = 300;
int shrink_factor = 4;
int density_factor = 3;

FILE *dvi_file;				/* user's file */

int font_not_found = 0;
struct font *current_font = NULL;	/* ptr into circular list of fonts */
#define MAX_OPEN_FONTS 12

/* Why so few? Because {OPEN_MAX} is the number of files that can be
 * left open; this may need some more adjusting though. */

int n_open_fonts = 0;			/* for LRU management of fonts */

/*
 * DVI preamble and postamble information.
 */
char job_id[300];
int total_pages, maxstack;
int current_page;
double fraction, conv;
long numerator, denominator, magnification;
#ifndef X10
int exposed=0;
#else X10

#endif X10
/*
 * Offset in DVI file of last page, set in read_postamble().
 */
long last_page_offset;

/*
 * Table of page offsets in DVI file, indexed by page number - 1.
 * Initialized in prepare_pages().
 */
long *page_offset;

/*
 * Cursor and mask for valid cursor
 */
#include "xdvi.h"
#include "xdvi_mask.h"

Window win;
#ifndef X10
int backpix;
GC highpix, forepix;
#else X10
int forepix, backpix, highpix;
#endif X10

long screen_w, screen_h, page_w, page_h;
long min_x, max_x, min_y, max_y, base_x, base_y;
long smin_x, smax_x, smin_y, smax_y;
int redisplay = 0;

unsigned long num();
long snum();

extern char reverse_byte[];
char *malloc(), *calloc(), *index();

int GXfunc;
int backwards = 0;

/********************************************/
/* This part of the file handles PK format  */
/********************************************/

#define PK_ID      89
#define PK_CMD_START 240
#define PK_X1     240
#define PK_X2     241
#define PK_X3     242
#define PK_X4     243
#define PK_Y      244
#define PK_POST   245
#define PK_NOOP   246
#define PK_PRE    247

#define Pxl1(fp) (one(fp->file))
#define Pxl2(fp) (two(fp->file))
#define Pxl4(fp) (four(fp->file))

#define FontRd1(fp) sone(fp->file)
#define FontRd2(fp) stwo(fp->file)
#define FontRd4(fp) sfour(fp->file)

int PK_flag_byte;
unsigned PK_input_byte;
int PK_nybble;
int PK_bitpos;
int PK_dyn_f;
int PK_repeat_count;


int
PK_get_nyb( fp )
     register struct font *fp;
{
  unsigned temp;
  if( PK_bitpos < 0 )
    {
      PK_input_byte = Pxl1( fp );
      PK_bitpos = 4;
    }
  temp = PK_input_byte >> PK_bitpos;
  PK_bitpos -= 4;
  return( temp & 0xf );
}


int
PK_packed_num( fp )
     register struct font *fp;
{
  int i,j,k;
  if( ( i = PK_get_nyb( fp ) ) == 0 )
    {
      do
	{
	  j = PK_get_nyb( fp );
	  i++;
	}
      while( j == 0 );
      while( i > 0 )
	{
	  j = (j << 4) + PK_get_nyb( fp );
	  i--;
	}
      return( j - 15 + ( (13 - PK_dyn_f) << 4 ) + PK_dyn_f );
    }
  else
    {
      if( i <= PK_dyn_f ) return( i );
      if( i < 14 ) return( ( (i - PK_dyn_f - 1) << 4 ) +
			  PK_get_nyb( fp ) + PK_dyn_f + 1 );
      if( i == 14 ) PK_repeat_count = PK_packed_num( fp );
      else PK_repeat_count = 1;
      return( PK_packed_num( fp ) );
    }
}


void
PK_skip_specials( fp )
     register struct font *fp;
{
  int i,j;
  do
    {
      PK_flag_byte = Pxl1(fp);
      if( PK_flag_byte >= PK_CMD_START )
	{
	  switch( PK_flag_byte )
	    {
	    case PK_X1 :
	    case PK_X2 :
	    case PK_X3 :
	    case PK_X4 :
	      {
		i = 0;
		for( j = PK_CMD_START; j <= PK_flag_byte; j++ )
		  i = (i*256) + Pxl1(fp);
		for( j = 1; j <= i; j++ )
		  (void) Pxl1(fp);
		break;
	      }
	    case PK_Y :
	      (void) Pxl4(fp);
	    case PK_POST :
	    case PK_NOOP :
	      break;
	    default :
	      error("Unexpected %d in PK file %s\n",
		      PK_flag_byte, fp->fontname );
	      break;
	    }
	}
    }
  while( PK_flag_byte != PK_POST && PK_flag_byte >= PK_CMD_START );
}


RdPkFont(fp)
register struct font *fp;
{
  int i, hppp, vppp, magnify;
  /* 
   * Read the header of a packed pixel file.
   */
  fseek(fp->file,0l,0); /* rewind the file, just in case */

  if (debug & DBG_PK)
    fprintf(stderr, "Reading header for packed pixel file %s\n",
	    fp->fontname);

  if (Pxl1(fp) != PK_PRE) 
    {
      error("file %s lacks preamble command\n",fp->fontname);
    }

  if (Pxl1(fp) != PK_ID)
    {
      error("file %s has wrong PK id\n",fp->fontname);
    }

  for( i = Pxl1(fp); i > 0; i-- )
    (void) Pxl1(fp);		/* Skip comment */
  
  fp->design = FontRd4(fp);
  (void) Pxl4(fp);		/* Skip checksum */
  hppp = FontRd4(fp);
  vppp = FontRd4(fp);
  if( debug && hppp != vppp )
    error("Warning: aspect ratio not 1:1 for font %s\n",
	    fp->fontname);
  /*fp->f_scale = (int)( (( (float) hppp * 72.27 ) / (float) 65536 ) + 0.5 );*/
  
  /*if (fp->f_scale == 0) fp->f_scale = 1000;*/
  PK_skip_specials( fp );
  fp->glyphaddr = ftell(fp->file) - 1;
}


enum PK_pre_type { Short, Xtended, Long };
#define PK_row_size 100
int PK_row[ PK_row_size ];
int PK_rowix;
short PK_turn_on;
int PK_power[ 33 ] =
{
         0x1,        0x2,        0x4,        0x8,
        0x10,       0x20,       0x40,       0x80,
       0x100,      0x200,      0x400,      0x800,
      0x1000,     0x2000,     0x4000,     0x8000,
     0x10000,    0x20000,    0x40000,    0x80000,
    0x100000,   0x200000,   0x400000,   0x800000,
   0x1000000,  0x2000000,  0x4000000,  0x8000000,
  0x10000000, 0x20000000, 0x40000000, 0x80000000
};

int PK_gpower[ 34 ] =
{
         0x0,        0x1,        0x3,        0x7,
         0xf,       0x1f,       0x3f,       0x7f,
        0xff,      0x1ff,      0x3ff,      0x7ff,
       0xfff,     0x1fff,     0x3fff,     0x7fff,
      0xffff,    0x1ffff,    0x3ffff,    0x7ffff,
     0xfffff,   0x1fffff,   0x3fffff,   0x7fffff,
    0xffffff,  0x1ffffff,  0x3ffffff,  0x7ffffff,
   0xfffffff, 0x1fffffff, 0x3fffffff, 0x7fffffff,
  0xffffffff
};
  

int
PK_read_char( fp, cp )
     register struct font *fp;
     char *cp;
{
  int bytes_left, cc, thispos, i, j;
  int row_bit_pos, row_mask;
  char row_byte;
  enum PK_pre_type pre_type;
  register struct glyph *g;
  long fpwidth;
  int word, word_weight, rwidth, width;
  int rows_left, h_bit, count;

  thispos = ftell( fp->file ) - 1; /* has already read flag byte */
  PK_dyn_f = PK_flag_byte >> 4;
  PK_flag_byte &= 0xf;
  PK_turn_on = ( PK_flag_byte >= 8 );
  PK_flag_byte &= 0x7;
  if( PK_flag_byte == 7 )
    {
      /* read first part of long character preamble */
      pre_type = Long;
      bytes_left = FontRd4( fp ) - 28;
      cc = FontRd4( fp );
      if( cc > 127 || cc < 0 )
	{
	  error("Character code %d outside range, file %s\n",
		  cc, fp->fontname );
	}
    }
  else
    if( PK_flag_byte > 3 )
      {
	/* read first part of extended short character preamble */
	pre_type = Xtended;
	bytes_left = ((PK_flag_byte - 4) << 16) + Pxl2( fp ) - 13;
	cc = Pxl1( fp );
	if( cc > 127 )
	  {
	    error("Character code %d outside range, file %s\n",
		    cc, fp->fontname );
	  }
      }
    else
      {
	/* read first part of short character preamble */
	pre_type = Short;
	bytes_left = (PK_flag_byte << 8) + Pxl1( fp ) - 8;
	cc = Pxl1( fp );
	if( cc > 127 )
	  {
	    error("Character code %d outside range, file %s\n",
		    cc, fp->fontname);
	  }
      }

  /* setup pointers for this char */
  g = fp->glyph + cc;
  g->addr = thispos;
  g->bitmap.bits = 0;		/* not in core -- may be altered below */

  if(debug & DBG_PK) {
    if( cp != NULL )
      fprintf( stderr, "loading pk " );
    else
      fprintf( stderr, "reading glyph data for pk " );
    fprintf( stderr, "char %d", cc );
    fprintf( stderr, ", char type " );
  }

  /* now read rest of character preamble */
  switch ( pre_type )
    {
    case Short :
      {
        if(debug & DBG_PK) fprintf( stderr, "short" );
	fpwidth = Pxl1( fp ) << 16;
	fpwidth += Pxl2( fp );
	(void) Pxl1( fp );	/* vertical escapement */
	g->bitmap.w = Pxl1( fp );
	g->bitmap.h = Pxl1( fp );
	g->x = FontRd1( fp );
	g->y = FontRd1( fp );
	break;
      }
    case Xtended :
      {
        if(debug & DBG_PK) fprintf( stderr, "extended" );
	fpwidth = Pxl1( fp ) << 16;
	fpwidth += Pxl2( fp );
	(void) Pxl2( fp );	/* vertical escapement */
	g->bitmap.w = Pxl2( fp );
	g->bitmap.h = Pxl2( fp );
	g->x = FontRd2( fp );
	g->y = FontRd2( fp );
	break;
      }
    case Long :
      {
	if(debug & DBG_PK) fprintf( stderr, "long" );
	fpwidth = FontRd4( fp );
	(void) FontRd4( fp );	/* horizontal escapement */
	(void) FontRd4( fp );	/* vertical escapement */
	g->bitmap.w = FontRd4( fp );
	g->bitmap.h = FontRd4( fp );
	g->x = FontRd4( fp );
	g->y = FontRd4( fp );
	if( g->bitmap.w < 0 || g->bitmap.h < 0
	   || g->bitmap.w > 0xffff || g->bitmap.h > 0xffff )
	  {
	    error("Too large character (%d) in file %s\n",
		    cc, fp->fontname);
	  }
	break;
      }
    } /* end of switch */
  g->x /= shrink_factor;
  g->y /= shrink_factor;

  g->dvi_adv = ((double) fp->scale * fpwidth) / (1 << 20);
  g->pxl_adv = pixel_round(g->dvi_adv);

  if ( g->bitmap.w != 0 )
    {
      if(debug & DBG_PK)
      {
        fprintf( stderr, ", size=%dx%d",g->bitmap.w,g->bitmap.h );
        fprintf( stderr, ", dvi_adv=%d pxl_adv=%d",g->dvi_adv,g->pxl_adv );
      }
    }
  if(debug & DBG_PK) fprintf( stderr, "\n" );

  if( cp != NULL && g->bitmap.w != 0 ) /* read character data into *cp */
    {
      g->bitmap.bits = cp;
      PK_bitpos = -1;
      if( PK_dyn_f == 14 )	/* get raster by bits */
	{
          rwidth = (g->bitmap.w + 15) / 16 * 2;
          bzero(g->bitmap.bits, g->bitmap.h * rwidth);
	  for (i = 0; i < g->bitmap.h; i++)  /* get all rows */
	    {
              cp = g->bitmap.bits + i * rwidth;
              row_bit_pos = 8;
	      for (j = 0; j < g->bitmap.w; j++)  /* get one row */
		{
                  if (--PK_bitpos < 0)
                    {
                      cc = Pxl1(fp);
                      PK_bitpos = 7;
                    }
                  if (--row_bit_pos < 0)
                    {
                      cp++;
                      row_bit_pos = 7;
                    }
                  if (cc & (1 << PK_bitpos))
                    *cp |= 1 << row_bit_pos;
                }
	    }
	}
      else
	{
	  /* get packed raster */
	  rows_left = g->bitmap.h;
	  h_bit = g->bitmap.w;
	  PK_repeat_count = 0;
	  word_weight = 32;
	  word = 0;
	  PK_rowix = 0;
	  while( rows_left > 0 )
	    {
	      count = PK_packed_num( fp );
	      while( count > 0 )
		{
		  if( count < word_weight && count < h_bit )
		    {
		      if( PK_turn_on )
			word += PK_gpower[ word_weight ]
			  - PK_gpower[ word_weight - count ];
		      h_bit -= count;
		      word_weight -= count;
		      count = 0;
		    }
		  else
		    if( count >= h_bit && h_bit <= word_weight )
		      {
			if( PK_turn_on )
			  word += PK_gpower[ word_weight ]
			    - PK_gpower[ word_weight - h_bit ];
			PK_row[ PK_rowix++ ] = word;
			/* "output" row(s) */
			rwidth = ((g->bitmap.w+15)/16)*2; /*## fix */
			for( i = 0; i <= PK_repeat_count; i++ )
			  {
			    for ( j = 0; j < rwidth; j++ )
			      {
				row_bit_pos = 24 - ( ( j & 0x3 ) << 3 );
				row_mask = 0xff << row_bit_pos;
				row_byte = (char) ( ( PK_row[ j >> 2 ] 
						     & row_mask )
						   >> row_bit_pos );
				*cp++ = row_byte;
			      }
			  }
			rows_left -= PK_repeat_count + 1;
			PK_repeat_count = 0;
			PK_rowix = 0;
			word = 0;
			word_weight = 32;
			count -= h_bit;
			h_bit = g->bitmap.w;
		      }
		    else
		      {
			if( PK_turn_on )
			  word |= PK_gpower[ word_weight ];
			PK_row[ PK_rowix++ ] = word;
			word = 0;
			count -= word_weight;
			h_bit -= word_weight;
			word_weight = 32;
		      }
		}
	      PK_turn_on = ( PK_turn_on ? 0 : 1 );
	    }
	  if( rows_left != 0 || h_bit != g->bitmap.w )
	    {
	      error("Bad pk file (%s), char %d, too many bits\n"
		      ,fp->fontname, cc );
	    }
	}
    }
  else
    {
      /* skip over data of character */
      for( i = 1; i <= bytes_left; i++ )
	(void) Pxl1( fp );
    }
}


RdPFont(fp)
register struct font *fp;
{
/* 
 * Read the header of a pixel file.
 */
	if (debug & DBG_PK)
		fprintf(stderr, "Reading header for pixel file %s\n",
		fp->fontname);

	if (Pxl4(fp) != 1001) {
	    error("file %s is ill-formed PXL file (1)\n",fp->fontname);
	}
	fseek(fp->file,(long)-(5*4),2);
	(void) Pxl4(fp);
	/*fp->f_scale = Pxl4(fp);*/
	/*if (fp->scale == 0) fp->scale = 1000;*/
	fp->design = Pxl4(fp);
	fp->glyphaddr = Pxl4(fp) * 4;  /* Pointer to glyph directory in file */
	if ((Pxl4(fp) != 1001)) {
    	    error("File %s is ill-formed PXL file (2)\n",fp->fontname);
	}
}


/********************************************/
/* end of PK routines                       */
/********************************************/


main(argc, argv)
	int argc;
	char **argv;
{
#ifndef X10
	int xargc=argc;
	char **xargv=argv;
#endif X10
	char *prog, *file;
	char *display = NULL;
	char *option;
#ifndef X10
	Frame frame;
#else X10
	OpaqueFrame frame;
#endif X10
	int reverse = 0;
	int bwidth = 2;
	char *fore_color;
	char *back_color;
	char *high_color;
	char *brdr_color;
	char *mous_color;
	char *geometry = NULL, def[32];
	int gflag;
#ifndef X10
	int bdrpix, mouspix;
	XColor cdef;
	XSizeHints hints;
	XSetWindowAttributes xswattrs;
	unsigned long xswattrs_mask;
#else X10
	int backmap, bdrmap, mouspix;
	Color cdef;
#endif X10
	int indian = 1;
	char *getenv();
#ifndef X10
	prog = argv[0];
#else
	if (*(char *) &indian) backwards = 0; else backwards = 1;
	/* x10 only, x11 swaps much differently. */
#endif X10
	prog = *argv++;
	argc--;
	/* prefetch the display argument, process the others later */
	while (argc) {
		if (strcmp(*argv, "-display") == 0 && argc > 1) {
			argv++;
			argc--;
			display = *argv;
		} else if (**argv == '=') {
			geometry = *argv;
		} else if (**argv != '-') {
			if (index(*argv, ':') != NULL)
				display = *argv;
			else
				file = *argv;
		}
		argv++;
		argc--;
	}
	argc = xargc;
	argv = xargv;
		
#ifndef X10
#define DPY DISP,
	if ((DISP = XOpenDisplay(display)) == NULL) {
#else X10
#define DPY
	  if (XOpenDisplay(display) == NULL) {
#endif X10
	    fprintf(stderr, "%s: Can't open display '%s'\n",
		    prog, XDisplayName(display));
	    exit(1);
	}
	if ((option = XGetDefault(DPY prog, "reverseVideo")) &&
	    strcmp(option, "on") == 0)
		reverse = 1;
	if (option = XGetDefault(DPY prog, "borderWidth"))
		bwidth = atoi(option);
	fore_color = XGetDefault(DPY prog, "foreground");
	back_color = XGetDefault(DPY prog, "background");
	high_color = XGetDefault(DPY prog, "highlight");
	brdr_color = XGetDefault(DPY prog, "borderColor");
	mous_color = XGetDefault(DPY prog, "pointerColor");
	file = NULL;
	if(!(font_path=getenv(FONTPATH)))
	  {
	    font_path=DEFAULT_FONT_PATH;
	  }
        while (argc) {
		if (strcmp(*argv, "-d") == 0) {
			argv++;
			argc--;
			debug = isdigit(argv[0][0]) ? atoi(argv[0]) : DBG_ALL;
		} else if (strcmp(*argv, "-l") == 0)
			list_fonts = 1;
		else if (strcmp(*argv, "-s") == 0 && argc > 1) {
			argv++;
			argc--;
			shrink_factor = atoi(*argv);
			if (shrink_factor <= 0) goto usage;
		} else if (strcmp(*argv, "-S") == 0 && argc > 1) {
			argv++;
			argc--;
			density_factor = atoi(*argv);
			if (density_factor < 0) goto usage;
		} else if (strcmp(*argv, "-p") == 0 && argc > 1) {
			argv++;
			argc--;
			pixels_per_inch = atoi(*argv);
			if (pixels_per_inch <= 0) goto usage;
		} else if (strcmp(*argv, "-rv") == 0) {
			reverse = 1;
		} else if (strcmp(*argv, "-fg") == 0 && argc > 1) {
			argv++;
			argc--;
			fore_color = *argv;
		} else if (strcmp(*argv, "-bg") == 0 && argc > 1) {
			argv++;
			argc--;
			back_color = *argv;
		} else if (strcmp(*argv, "-hl") == 0 && argc > 1) {
			argv++;
			argc--;
			high_color = *argv;
		} else if (strcmp(*argv, "-bd") == 0 && argc > 1) {
			argv++;
			argc--;
			brdr_color = *argv;
		} else if (strcmp(*argv, "-ms") == 0 && argc > 1) {
			argv++;
			argc--;
			mous_color = *argv;
		} else if (strcmp(*argv, "-geometry") == 0 && argc > 1) {
			argv++;
			argc--;
			geometry = *argv;
		} else if (strcmp(*argv, "-display") == 0 && argc > 1) {
			argv++;
			argc--;
			display = *argv; /* already set and used */
		} else if (**argv == '=') {
			geometry = *argv;
		} else if (**argv != '-') {
			if (index(*argv, ':') != NULL)
				display = *argv; /* already set and used */
			else
				file = *argv;
		} else {
		    usage:
			fprintf(stderr, "\
Usage: xdvi [-s <shrink>] [-S <density>] [-p <pixels>] [-l] [-rv]\n\
       [-fg <color>] [-bg <color>] [-hl <color>] [-bd <color>] [-ms <color>]\n\
       [-geometry <geometry> | =<geometry>]\n\
       [-display <host:display> | host:display] dvi_file\n");
			exit(1);
		}
		argv++;
		argc--;
	}
        if (file == NULL)
		goto usage;
	if ((dvi_file = fopen(file, "r")) == NULL) {
		int n = strlen(file);
		char *dvi_name;

		if (strcmp(file + n - sizeof(".dvi") + 1, ".dvi") == 0) {
			perror(file);
			exit(1);
		}
		dvi_name = malloc((unsigned) n + sizeof(".dvi"));
		sprintf(dvi_name, "%s.dvi", file);
		if ((dvi_file = fopen(dvi_name, "r")) == NULL) {
			perror(dvi_name);
			exit(1);
		}
 	}
	process_preamble();
	find_postamble();
	read_postamble();
	prepare_pages();
	init_page();

#ifndef X10
	SCRN = DefaultScreen(DISP);  
	highpix = XCreateGC(DISP, RootWindow(DISP, SCRN), 0, NULL);
	forepix = XCreateGC(DISP, RootWindow(DISP, SCRN), 0, NULL);
	XCopyGC(DISP, DefaultGC(DISP, SCRN), (1L<<(GCLastBit+1))-1, highpix);
	XCopyGC(DISP, DefaultGC(DISP, SCRN), (1L<<(GCLastBit+1))-1, forepix);

	/* forepix is for lines, *not* text, *not* image */
	XSetForeground(DISP, forepix,
		       WhitePixel(DISP, SCRN)^BlackPixel(DISP, SCRN));
	XSetFunction(DISP, forepix, GXxor);

	/* highpix is for text and images (not lines) */
	XSetForeground(DISP, highpix,
		       reverse?WhitePixel(DISP,SCRN):BlackPixel(DISP,SCRN));
	XSetFunction(DISP, highpix, GXcopy); /* ignored... */
#else X10
	{
		int xorpix = WhitePixel^BlackPixel;
		forepix = xorpix;
		highpix = xorpix;
		GXfunc = GXxor;
	}
#endif X10
	if (reverse) {
#ifndef X10
		backpix = BlackPixel(DISP, SCRN);
		bdrpix  = WhitePixel(DISP, SCRN);
		mouspix = WhitePixel(DISP, SCRN);
#else X10
		backpix = BlackPixel;
		backmap = BlackPixmap;
		bdrmap = WhitePixmap;
		mouspix = WhitePixel;
#endif X10
	} else {
#ifndef X10
		backpix = WhitePixel(DISP, SCRN);
		bdrpix  = BlackPixel(DISP, SCRN);
		mouspix = BlackPixel(DISP, SCRN);
#else X10
		backpix = WhitePixel;
		backmap = WhitePixmap;
		bdrmap = BlackPixmap;
		mouspix = BlackPixel;
#endif X10
	}
#ifndef X10
#define XPC(col, cd) XParseColor(DISP, DefaultColormap(DISP, SCRN), col, cd)
#define XGHC(cd) XAllocColor(DISP, DefaultColormap(DISP, SCRN), cd)
	if (DisplayCells(DISP, SCRN) > 2) {
		unsigned long cfore;
		cfore = reverse?WhitePixel(DISP, SCRN):BlackPixel(DISP,SCRN);
		backpix = reverse?BlackPixel(DISP, SCRN):WhitePixel(DISP,SCRN);
		if (fore_color && XPC(fore_color, &cdef) &&
			XGHC(&cdef))
			cfore = cdef.pixel;
		if (back_color && XPC(back_color, &cdef) &&
			XGHC(&cdef)) {
#else X10
	if (DisplayCells() > 2) {
		if (fore_color && XParseColor(fore_color, &cdef) &&
			XGetHardwareColor(&cdef))
			forepix = cdef.pixel;
		if (back_color && XParseColor(back_color, &cdef) &&
			XGetHardwareColor(&cdef)) {
#endif X10
			backpix = cdef.pixel;
#ifdef X10
			backmap = XMakeTile(backpix);
#endif X10
		}
#ifndef X10
		XSetForeground(DISP, highpix, cfore);
		XSetForeground(DISP, forepix, cfore^backpix);
		if (high_color && XPC(high_color, &cdef) &&
			XGHC(&cdef))
			XSetForeground(DISP, highpix, cdef.pixel);
		if (brdr_color && XPC(brdr_color, &cdef) &&
			XGHC(&cdef));
		if (mous_color && XPC(mous_color, &cdef) &&
			XGHC(&cdef))
#else X10
		if (high_color && XParseColor(high_color, &cdef) &&
			XGetHardwareColor(&cdef))
			highpix = cdef.pixel;
		if (brdr_color && XParseColor(brdr_color, &cdef) &&
			XGetHardwareColor(&cdef))
			bdrmap = XMakeTile(cdef.pixel);
		if (mous_color && XParseColor(mous_color, &cdef) &&
			XGetHardwareColor(&cdef))
#endif X10
			mouspix = cdef.pixel;
	}
#ifndef X10
	XSetBackground(DISP, forepix, backpix);
	XSetBackground(DISP, highpix, backpix);
#endif X10
	frame.bdrwidth = bwidth;
	frame.height = page_h;
#ifndef X10
	if (frame.height + (bwidth << 1) > DisplayHeight(DISP, SCRN))
	    frame.height = DisplayHeight(DISP, SCRN) - (bwidth << 1);
#else X10
	if (frame.height + (bwidth << 1) > DisplayHeight())
	    frame.height = DisplayHeight() - (bwidth << 1);
#endif X10
	frame.width = page_w;
#ifndef X10
	if (frame.width + (bwidth << 1) > DisplayWidth(DISP, SCRN))
	    frame.width = DisplayWidth(DISP, SCRN) - (bwidth << 1);
 	frame.border = bdrpix;
	/* 	frame.background = backmap; */
#else X10
	if (frame.width + (bwidth << 1) > DisplayWidth())
	    frame.width = DisplayWidth() - (bwidth << 1);
	frame.border = bdrmap;
	frame.background = backmap;
#endif X10
	frame.x = 0;
	frame.y = 0;
	sprintf(def, "=%dx%d+0+0", frame.width, frame.height);
#ifndef X10

	hints.width = page_w;
	hints.height = page_h;
	hints.x = 0; hints.y = 0;
	hints.flags = (PSize | PPosition);
	
	xswattrs.border_pixel = bdrpix;
	xswattrs.background_pixel = backpix;
	xswattrs_mask = (CWBackPixel|CWBorderPixel);

	win = XCreateWindow(DISP, RootWindow(DISP, SCRN),
			    frame.x, frame.y,
			    frame.width, frame.height,
			    frame.bdrwidth,
			    0,	/* depth from parent */
			    InputOutput,
			    CopyFromParent,
			    xswattrs_mask,
			    &xswattrs);

	XSetStandardProperties(DISP, win,
			       prog, "DVI Previewer",
			       None,
			       xargv, xargc, &hints);

	gflag = XParseGeometry(geometry, &hints.x, &hints.y,
			       &hints.width, &hints.height);
			       
	hints.flags =
	  ((gflag&(XValue|YValue))?(USPosition):0) |
	    ((gflag&(WidthValue|HeightValue))?(USSize):0);
	
	XSetNormalHints(DISP, win, &hints);
	
	exposed = 0;		/* make sure we paint ONCE */
#else X10
	win = XCreate("DVI Previewer", prog, geometry, def, &frame, 50, 50);
#endif X10
	screen_h = frame.height;
	screen_w = frame.width;
#ifndef X10
	XMapWindow(DISP, win);
	XSelectInput(DISP, win,
		     KeyPressMask|ButtonPressMask|ExposureMask);
#ifdef POSTPS
	XDefineCursor(DISP, win,
		      XCreateFontCursor(DISP, XC_gumby));
#else /* POSTPS */
	XDefineCursor(DISP, win,
		      XCreateFontCursor(DISP, XC_cross));
#endif /* POSTPS */
	XFlush(DISP);
#else X10
	XMapWindow(win);
	XSelectInput(win, KeyPressed|ButtonPressed|ExposeWindow|ExposeRegion);
	XDefineCursor(win,
	    XCreateCursor(xdvi_width, xdvi_height, xdvi_bits, xdvi_mask_bits,
			  xdvi_x_hot, xdvi_y_hot, mouspix, backpix, GXcopy));
#endif X10
	do_pages();
	stop_output(0);
}

/*
**      process_preamble reads the information in the preamble and stores
**      it into global variables for later use.
*/
process_preamble()
{
        ubyte   k;

        if (one(dvi_file) != PRE)
		error("xdvi: DVI file doesn't start with preamble");
	if (one(dvi_file) != 2)
		error("xdvi: Wrong version of DVI output for this program");
	numerator     = four(dvi_file);
	denominator   = four(dvi_file);
	magnification = four(dvi_file);
	fraction = (((double) numerator * magnification)
	                                 / ((double) denominator * 1000.));
	define_conv();
	k = one(dvi_file);
	fread(job_id, sizeof(char), k, dvi_file);
	job_id[k] = '\0';
}

define_conv ()
{
	conv = ((fraction * pixels_per_inch) / 100000) / (2.54 * shrink_factor);
}

/*
**      find_postamble locates the beginning of the postamble
**	and leaves the file ready to start reading at that location.
*/
find_postamble()
{
	ubyte byte;
	long offset = -4;        /* At least 4 TRAILERS */

	do {
		offset -= 1;
		fseek(dvi_file, offset, 2);
		byte = one(dvi_file);
	} while (byte == TRAILER);
	if (byte != 2)
		error("xdvi: Wrong version of DVI output for this program");
	offset -= 4;
	fseek(dvi_file, offset, 2);
	fseek(dvi_file, sfour(dvi_file), 0);
}

/*
**      read_postamble reads the information in the postamble,
**	storing it into global variables.
**      It also takes care of reading in all of the PXL files for the fonts
**      used in the job.
*/
read_postamble()
{
        ubyte   cmnd;
	int page_width, page_height;

        if (one(dvi_file) != POST)
	    error("xdvi: Postamble doesn't begin with POST");
	last_page_offset = four(dvi_file);
	if (numerator != four(dvi_file)
	          ||  denominator != four(dvi_file)
		  ||  magnification != four(dvi_file))
	    error("xdvi: Postamble doesn't match preamble");
	page_height = pixel_round(four(dvi_file));
	page_width = pixel_round(four(dvi_file));
	maxstack = two(dvi_file);
	total_pages = two(dvi_file);
	do {
	    switch(cmnd = one(dvi_file)) {
	        case FNTDEF1:
	        case FNTDEF2:
	        case FNTDEF3:
	        case FNTDEF4:
		    define_font(cmnd);
		    break;
		case POSTPOST:
		    break;
		default:
		    error("xdvi: Non-fntdef cmnd found in postamble");
	    }
	} while (cmnd != POSTPOST);
	if (font_not_found)
		error("xdvi: Not all PXL files were found");
	list_fonts = 0;
}

prepare_pages()
{
	int i;

        stack = (struct frame *) malloc((unsigned) sizeof(struct frame) * (maxstack+1));
        if (stack == NULL)
		error("xdvi: Can't allocate stack space (%d frames)", maxstack);
	page_offset = (long *) malloc((unsigned) total_pages * sizeof(long));
        if (page_offset == NULL)
		error("xdvi: Can't allocate page directory (%d pages)", total_pages);
	i = total_pages;
	page_offset[--i] = last_page_offset;
	fseek(dvi_file, last_page_offset, 0);
	/*
	 * Follow back pointers through pages in the DVI file,
	 * storing the offsets in the page_offset table.
	 */
	while (i > 0) {
		num(dvi_file, 1+4+(9*4));
		fseek(dvi_file, page_offset[--i] = four(dvi_file), 0);
	}
}

/*
**      define_font reads the rest of the fntdef command and then reads in
**      the specified PXL file, adding it to the global linked-list holding
**      all of the fonts used in the job.
*/
define_font(cmnd)
	ubyte cmnd;
{
        register struct font *fontp;
	int len;
	int unmodsize;
	float realsize;
	int size,pksize;
        long checksum;

	fontp = (struct font *) malloc((unsigned) sizeof(struct font));
	if (fontp == NULL)
		error("xdvi: Can't allocate memory for font");
	fontp->TeXnumber = num(dvi_file, cmnd - FNTDEF1 + 1);
	checksum = four(dvi_file);
	fontp->scale = four(dvi_file);
	fontp->design = four(dvi_file);
	len = one(dvi_file) + one(dvi_file);
	fontp->fontname = malloc(len + 10);	/* leave space for magnification */
	fread(fontp->fontname, sizeof(char), len, dvi_file);
	fontp->fontname[len] = '\0';
	fontp->file = NULL;
	if(debug & DBG_PK)
	  fprintf(stderr,"Define font \"%s\" scale=%d design=%d\n",
		  fontp->fontname,fontp->scale,fontp->design);
/*
**	In the actual implementation, scaled-size/design-size hasn't been
**	stored with sufficient precision, hence the messing around to find
**	its actual value.
*/
	realsize = (magnification/1000.)*((float) fontp->scale / fontp->design);
	unmodsize = (realsize * 1000) + 0.5;
	/* a real hack to correct for rounding in some cases */
	switch (unmodsize) {
	    case 1095:
		realsize = 1.095445;	/* stephalf */
		break;
	    case 1315:
		realsize = 1.314534;	/* stepihalf */
		break;
	    case 2074:
		realsize = 2.0736;	/* stepiv */
		break;
	    case 2488:
		realsize = 2.48832;	/* stepv */
		break;
	    case 2986:
		realsize = 2.985984;	/* stepiv */
		break;
	}
	/*
	 * the remaining magnification steps are represented
	 * with sufficient accuracy already
	 */
	size = (realsize * pixels_per_inch * 5) + 0.5;
	pksize = (realsize * pixels_per_inch) + 0.5;
	sprintf(&fontp->fontname[len], FONT_SUFFIX_PK, pksize);
	if (debug & DBG_PK) fprintf(stderr,"PKfilename=<%s>",fontp->fontname);
	if(open_pxl_file(fontp)) { /* THIS DOESN'T WORK QUITE RIGHT */
				/* since the open guesses on what it gets
				 * and ignores the extension... make this
				 * a site configureable option? Assume
				 * EITHER pk OR pxl??? [mwe/rlk] */
	  fontp->format = PkFormat;
	  read_glyphs(fontp);
	}
	else {
	  sprintf(&fontp->fontname[len], FONT_SUFFIX_PXL, size);
	  if (debug & DBG_PK) fprintf(stderr,"PXLfilename=<%s>",fontp->fontname);
	  if(open_pxl_file(fontp)) {
	    fontp->format = PxlFormat;
	    read_glyphs(fontp);
	  }
	  else {
	    fprintf(stderr,"%s [not found]\n",fontp->fontname);
	    font_not_found = 1;
	    return;
	  }
	}
	if (current_font == NULL) {
		fontp->next = fontp;
		fontp->prev = fontp;
	}
	else {
		fontp->next = current_font;
		fontp->prev = current_font->prev;
		current_font->prev->next = fontp;
		current_font->prev = fontp;
	}
	current_font = fontp;
}

open_pxl_file(font)
	struct font *font;
{
	char filename[300];
	extern int errno;
	if (font->file == NULL) {
		sprintf(filename, "%s/%s", /* note: UNUSED */
				FONT_DIRECTORY, font->fontname);
		if (n_open_fonts == MAX_OPEN_FONTS)
			close_lru();
		font->file = fopen_font_path(font_path, font->fontname, "r");
		if (font->file == NULL) {
		        if (list_fonts)
                                fprintf(stderr,"%s [missing]\n",font->fontname);
			return (0);
		}
		n_open_fonts += 1;
	}
	if (list_fonts)
		fprintf(stderr,"%s\n", font->fontname);
	return (1);
}

read_pxl_bitmap(ch, g)
	ubyte ch;
	register struct glyph *g;
{
	register struct bitmap *bitmap;
	register int file_bytes_wide;
	register char *ptr,*endbit;
	register int i, j;
	int bitmap_size;

	bitmap = &g->bitmap;

	/* in file, bitmap rows are multiples of 32 bits wide */
	file_bytes_wide = ROUNDUP(bitmap->w, BITS_PER_LONG)*BYTES_PER_LONG;
	/* width must be multiple of 16 bits for raster_op */
	bitmap->bytes_wide=ROUNDUP(bitmap->w, BITS_PER_SHORT)*BYTES_PER_SHORT;
	bitmap_size= (unsigned) bitmap->h * bitmap->bytes_wide;
	ptr = bitmap->bits = malloc(bitmap_size);
	if (ptr == NULL)
		error("xdvi: Can't allocate bitmap for character %d of font %s (%d by %d)",
			ch, current_font->fontname, bitmap->h, bitmap->w);
	if (!open_pxl_file(current_font))
		error("xdvi: Can't find font file %s", current_font->fontname);
	fseek(current_font->file, g->addr, 0);
	switch(current_font->format) {
	case PxlFormat:
		for (i = 0; i < bitmap->h; i += 1)
			for (j = 0; j < file_bytes_wide; j += 1)
				if (j < bitmap->bytes_wide)
					*ptr++ = reverse_byte[one(current_font->file)];
				else
					one(current_font->file);
		break;
	case PkFormat:
		PK_skip_specials(current_font);
		PK_read_char(current_font,bitmap->bits);
		endbit=bitmap->bits+bitmap_size;
		for (ptr=bitmap->bits; ptr<endbit; ++ptr)
			*ptr = reverse_byte[(*ptr & 0377)];
		break;
	default:
		error("Internal error, bat format\n");
	}
	if (shrink_factor != 1)
		shrink_bitmap(bitmap, shrink_factor, shrink_factor);
	if (debug & DBG_BITMAP)
		print_char(ch, g);
}

/*
 * Find font #n and move it to the head of the list.
 */
change_font(n)
	unsigned long n;
{
        register struct font *fontp;

	fontp = current_font;
	for (;;) {
		if (fontp->TeXnumber == n)
                        break;
		fontp = fontp->next;
		if (fontp == current_font)
			error("xdvi: Non-existent font #%d", n);
	}
	if (current_font == fontp)
		return;
	fontp->prev->next = fontp->next;
	fontp->next->prev = fontp->prev;
	fontp->next = current_font;
	fontp->prev = current_font->prev;
	current_font->prev->next = fontp;
	current_font->prev = fontp;
	current_font = fontp;
}

/*
 * Close the PXL file for the least recently used font.
 */
close_lru()
{
        register struct font *f;

	f = current_font->prev;
	for (;;) {
		if (f->file != NULL)
                        break;
		f = f->prev;
		if (f == current_font->prev)
			error("xdvi: Can't find an open PXL file to close");
	}
	fclose(f->file);
	f->file = NULL;
	n_open_fonts -= 1;
}

reset_fonts()
{
        register struct font *f;
	register struct glyph *g;

	f = current_font;
	for (;;) {
	    open_pxl_file(f);
	    for (g = &f->glyph[0]; g < &f->glyph[MAXCHARS]; g += 1) {
		if (g->bitmap.bits) free(g->bitmap.bits);
	    }
	    read_glyphs(f);
	    f = f->next;
	    if (f == current_font) break;
	}
}

read_P_glyphs (fontp)
        register struct font *fontp;
{
	register struct glyph *g;
        long checksum, magnify, design_size, font_dir_ptr, pxl_id_word;

	/* seek to trailer info */
	fseek(fontp->file, (long) -(5 * BYTES_PER_LONG), 2);
        checksum = four(fontp->file);
        magnify = four(fontp->file);
        design_size = four(fontp->file);
        font_dir_ptr = sfour(fontp->file) * 4;
        pxl_id_word = four(fontp->file);
#ifdef lint
	magnify = design_size = pxl_id_word = magnify;
#endif
	/* seek to font directory */
	fseek(fontp->file, font_dir_ptr, 0);
	for (g = &fontp->glyph[0]; g < &fontp->glyph[MAXCHARS]; g += 1) {
		g->bitmap.bits = NULL;
		g->bitmap.w = two(fontp->file);	/* leave this for shrink_bitmap */
		g->bitmap.h = two(fontp->file);	/* leave this for shrink_bitmap */
		g->x = stwo(fontp->file) / shrink_factor;
		g->y = stwo(fontp->file) / shrink_factor;
		g->addr = four(fontp->file) * 4;
		/*
		**  The TFM-width word is kind of funny in the units
		**  it is expressed in.  It works this way:
		**
		**  If a glyph has width 'w' in a font with design-size
		**  'd' (both in same units), the TFM-width word is
		**
		**                    (w/d) * 2^20
		**
		**  Therefore, in order to find the glyph width in
		**  DVI units (1 / 2^16 points), we take the design-size
		**  'd' (in DVI's), the magnification 'm' of the PXL file
		**  and the TFM-width word 't' to the width (in DVI's)
		**  as follows:
		**
		**                     dmt
		**                w = -----
		**                    2^20
		**
		**  But the magnification of the PXL file is just the
		**  scaled size 's' over the design size, so the final
		**  expression for the width is
		**
		**                     st
		**                w = ----
		**                    2^20
		**
		*/      

		g->dvi_adv =
			((double) fontp->scale * four(fontp->file)) / (1 << 20);
		g->pxl_adv = pixel_round(g->dvi_adv);
#ifdef VERBOSE_DEBUG
		fprintf(stderr,"char %d : dvi_adv=%d pxl_adv=%d\n",
		g - &fontp->glyph[0],g->dvi_adv,g->pxl_adv );
#endif
	}
}

read_PK_glyphs(fontp)
        register struct font *fontp;
{
	register struct glyph *g;
        long checksum, magnify, design_size, font_dir_ptr, pxl_id_word;

	RdPkFont(fontp);
	fseek(fontp->file,fontp->glyphaddr,0);
	PK_skip_specials( fontp );
	while( PK_flag_byte != PK_POST )
	  {
	    PK_read_char(fontp,(char *)0);	/* read "glyph directory" */
	    PK_skip_specials(fontp); /* (really a whole pass over file) */
	  }
}

read_glyphs(fontp)
        register struct font *fontp;
{
  switch(fontp->format)
    {
    case PxlFormat:
      read_P_glyphs(fontp);
      break;
    case PkFormat:
      read_PK_glyphs(fontp);
      break;
    }
}


#define nope(str)       error("xdvi: %s not implemented", str)
#define correct()       (PXL_H = pixel_round(DVI_H))

do_pages()
{
        ubyte ch;

	min_x = 0;
	min_y = 0;
	max_x = screen_w;
	max_y = screen_h;
	base_x = min_x;
	base_y = min_y;
	current_page = 0;
	for (;;) {
		ch = one(dvi_file);
		if (debug & DBG_DVI)
			print_dvi(ch);
		if (ch <= SETCHAR0 + 127) {
			set_char(ch);
			DVI_H += current_font->glyph[ch].dvi_adv;
			PXL_H += current_font->glyph[ch].pxl_adv;
			correct();
		} else if (FNTNUM0 <= ch  &&  ch <= FNTNUM0 + 63) {
			change_font((unsigned long) (ch - FNTNUM0));
		} else {
			long a, b;

			switch (ch) {
			    case SET1:
				nope("SET1");
				break;

			    case SETRULE:
				a = sfour(dvi_file); b = sfour(dvi_file);
				if (a > 0  &&  b > 0) {
				    correct();
				    set_rule(pixel_round(a), pixel_round(b));
				}
				DVI_H += b;
				PXL_H =  pixel_round(DVI_H);
				break;

			    case PUT1:
				nope("PUT1");
				break;

			    case PUTRULE:
				a = sfour(dvi_file); b = sfour(dvi_file);
				if (a > 0  &&  b > 0) {
				    correct();
				    set_rule(pixel_round(a), pixel_round(b));
				}
				break;

			    case NOP:
				break;

			    case BOP:
				num(dvi_file, 11*4);
				stackp = 0;
				DVI_H = dvi_round(X_PAGE_OFFSET);
				PXL_H = X_PAGE_OFFSET;
				DVI_V = dvi_round(Y_PAGE_OFFSET);
				PXL_V = Y_PAGE_OFFSET;
				WW = XX = YY = ZZ = 0;
				begin_page();
				break;

			    case EOP:
				if (stackp > 0)
				    error("Stack not empty at EOP (%d)",
				    	   stackp);
				end_page();
				if (ftell(dvi_file) > last_page_offset)
				    return;
				break;

			    case PUSH:
				stackp++;
				if (stackp > maxstack)
				    error("xdvi: More PUSHes than were promised");
				stack[stackp] = stack[stackp - 1];
				break;

			    case POP:
				stackp--;
				if (stackp < 0)
				    error("xdvi: More POPs than PUSHes");
				break;

			    case RIGHT1:
			    case RIGHT2:
			    case RIGHT3:
			    case RIGHT4:
				DVI_H += snum(dvi_file, ch - RIGHT1 + 1);
				PXL_H = pixel_round(DVI_H);
				break;

			    case X0:
			    case X1:
			    case X2:
			    case X3:
			    case X4:
				if (ch != X0)
				    XX = snum(dvi_file, ch - X0);
				DVI_H += XX;
				PXL_H += pixel_round(XX);
				correct();
				break;

			    case W0:
			    case W1:
			    case W2:
			    case W3:
			    case W4:
				if (ch != W0)
				    WW = snum(dvi_file, ch - W0);
				DVI_H += WW;
				PXL_H = pixel_round(DVI_H);
				break;

			    case Y0:
			    case Y1:
			    case Y2:
			    case Y3:
			    case Y4:
				if (ch != Y0)
				    YY = snum(dvi_file, ch - Y0);
				DVI_V += YY;
				PXL_V = pixel_round(DVI_V);
				break;

			    case Z0:
			    case Z1:
			    case Z2:
			    case Z3:
			    case Z4:
				if (ch != Z0)
				    ZZ = snum(dvi_file, ch - Z0);
				DVI_V += ZZ;
				PXL_V = pixel_round(DVI_V);
				break;

			    case DOWN1:
			    case DOWN2:
			    case DOWN3:
			    case DOWN4:
				DVI_V += snum(dvi_file, ch - DOWN1 + 1);
				PXL_V = pixel_round(DVI_V);
				break;

			    case FNT1:
			    case FNT2:
			    case FNT3:
			    case FNT4:
				change_font(num(dvi_file, ch - FNT1 + 1));
				break;

			    case XXX1:
			    case XXX2:
			    case XXX3:
			    case XXX4:
				a = num(dvi_file, ch - XXX1 + 1);
				if(a > 0)
				    special((unsigned long) a);
				break;

			    case FNTDEF1:
			    case FNTDEF2:
			    case FNTDEF3:
			    case FNTDEF4:
				fseek(dvi_file, (long) (12 + ch - FNTDEF1 + 1), 1);
				a = one(dvi_file) + one(dvi_file);
				fseek(dvi_file, (long) a, 1);
				break;

			    case PRE:
				error("xdvi: Shouldn't happen: PRE encountered.");
				break;

			    case POST:
				error("xdvi: Shouldn't happen: POST encountered.");
				break;

			    case POSTPOST:
				error("xdvi: Shouldn't happen: POSTPOST encountered.");
				break;

			    default:
				error("xdvi: Unknown op-code %d, offset %d",
					ch, ftell(dvi_file));
			} /* end switch*/
		} /* end else (ch not a SETCHAR or FNTNUM) */
	} /* end for */
}

set_char(ch)
	ubyte ch;
{
	register struct glyph *g;

	g = &current_font->glyph[ch];
	if (g->bitmap.bits == NULL) {
		read_pxl_bitmap(ch, g);
		if (backwards) reverse_bytes(&g->bitmap);
	      }
	
	put_bitmap(&g->bitmap, (PXL_H - g->x), (PXL_V - g->y), highpix);
}

reverse_bytes(bitmap)
 	register struct bitmap *bitmap;
{
 	register long x, y;
 	register char *bp ;
 	register char c ;
 
 	bp = bitmap->bits ;
 	for ( y = 0 ; y < bitmap->h ; y++) {
 		for ( x = 0 ; x < bitmap->bytes_wide ; x += 2) {
 			c = *bp ;
 			*bp = *(bp + 1) ;
			bp++;
 			*bp++ = c ;
 		}
 	}
}
 
set_rule(h, w)
	long h, w;
{
	/* (w,h) specifies lower left corner of rule box */
	put_rectangle(PXL_H, PXL_V - h, w, h, forepix);
}

begin_page()
{
	if (debug & DBG_BATCH)
		return;
	if (!redisplay)
	    clear_page();
	put_border(0, 0, page_w, page_h, 1);
}

#ifndef X10
#define TRSIZE 100
#endif X10
end_page()
{
	int ch, arg, sign, number, next_page;
	XEvent event;
#ifndef X10
	char trbuf[TRSIZE];
	XComposeStatus kst;
#endif X10
	char *string;
#ifndef X10
	int nbytes=0;
#else X10
	int nbytes;
#endif X10

#ifdef lint
	number = 0;
#endif
	if (debug & DBG_BATCH) {
		if (++current_page == total_pages)
			exit(0);
		return;
	}
	if (redisplay) {
	    min_x = smin_x;
	    max_x = smax_x;
	    min_y = smin_y;
	    max_y = smax_y;
	    redisplay = 0;
	}
	arg = 0;
	for (;;) {
#ifndef X10
		XNextEvent (DISP, &event);
#else X10
		XNextEvent (&event);
#endif X10
		switch (event.type) {
#ifndef X10
		case Expose:
		    exposed = 1;
#else X10
		case ExposeWindow:
		    screen_h = ((XExposeEvent *)(&event))->height;
		    screen_w = ((XExposeEvent *)(&event))->width;
		    max_x = min_x + screen_w;
		    max_y = min_y + screen_h;
		    string = "\f";
		    nbytes = 1;
		    break;
		case ExposeRegion:
#endif X10
		    smin_x = min_x;
		    smax_x = max_x;
		    smin_y = min_y;
		    smax_y = max_y;
#ifndef X10
		    min_x = min_x + event.xexpose.x;
		    min_y = min_y + event.xexpose.y;
		    max_x = min_x + event.xexpose.width;
		    max_y = min_y + event.xexpose.height;
#else X10
		    min_x = min_x + ((XExposeEvent *)(&event))->x;
		    min_y = min_y + ((XExposeEvent *)(&event))->y;
		    max_x = min_x + ((XExposeEvent *)(&event))->width;
		    max_y = min_y + ((XExposeEvent *)(&event))->height;
#endif X10
		    redisplay = 1;
		    string = "\f";
		    nbytes = 1;
		    break;
#ifndef X10

		case ButtonPress:
#else X10
		case ButtonPressed:
#endif X10
		    {
#ifndef X10
		    short detail = event.xbutton.button;
		    short state = event.xbutton.state;
		    switch (detail) {
		    case Button1:
			if (state & ShiftMask)
#else X10
		    short detail = ((XButtonPressedEvent *) (&event))->detail;
		    switch (detail & ValueMask) {
		    case LeftButton:
			if (detail & ShiftMask)
#endif X10
			    string = "l";
			else
			    string = "b";
			nbytes = 1;
			break;

#ifndef X10
		    case Button2:
#else X10
		    case MiddleButton:
#endif X10
/* special code for "exit on middle button" for PostPs emulation. */
#ifdef POSTPS
			string = "q";
#else /* POSTPS */
#ifndef X10
			if (state & ShiftMask)
#else X10
			if (detail & ShiftMask)
#endif X10
			    string = "u";
			else
			    string = "d";
#endif /* POSTPS */
			nbytes = 1;
			break;
#ifndef X10
		    case Button3:
			if (state & ShiftMask)
#else X10
		    case RightButton:
			if (detail & ShiftMask)
#endif X10
			    string = "r";
			else
			    string = "f";
			nbytes = 1;
			break;
		    }
		    }
		    break;
#ifndef X10
		case KeyPress:
		    string = trbuf;
		    nbytes = XLookupString(&event, string, TRSIZE, NULL, NULL);
#else X10
		case KeyPressed:
		    string = XLookupMapping (&event, &nbytes);
#endif X10
		    break;
		}
		if (nbytes == 0) continue;
		if (nbytes > 1) goto bad;
		switch (ch = *string) {
		    case 'q':
		    case '\003':	/* control-C */
		    case '\004':	/* control-D */
			stop_output(0);
			break;
		    case 'n':
		    case 'f':
		    case ' ':
			/* scroll forward */
			min_x = 0;
			min_y = 0;
			base_x = base_y = 0;
			max_x = screen_w;
			max_y = screen_h;
			next_page = current_page + 1;
			break;
		    case 'p':
		    case 'b':
		    case '\b':
			/* scroll backward */
			min_x = 0;
			min_y = 0;
			base_x = base_y = 0;
			max_x = screen_w;
			max_y = screen_h;
			next_page = current_page - 1;
			break;
		    case 'u':
			if (min_y == 0) goto bad;
			min_y -= screen_h;
			if (min_y < 0)
			    min_y = 0;
			base_y = min_y;
			max_y = min_y + screen_h;
			next_page = current_page;
			break;
		    case 'd':
			if (min_y >= page_h - screen_h) goto bad;
			min_y += screen_h;
			if (min_y > page_h - screen_h)
			    min_y = page_h - screen_h;
			if (min_y < 0)
			    min_y = 0;
			base_y = min_y;
			max_y = min_y + screen_h;
			next_page = current_page;
			break;
		    case 'l':
			if (min_x == 0) goto bad;
			min_x -= screen_w;
			if (min_x < 0)
			    min_x = 0;
			base_x = min_x;
			max_x = min_x + screen_w;
			next_page = current_page;
			break;
		    case 'r':
			if (min_x >= page_w - screen_w) goto bad;
			min_x += screen_w;
			if (min_x > page_w - screen_w)
			    min_x = page_w - screen_w;
			if (min_x < 0)
			    min_x = 0;
			base_x = min_x;
			max_x = min_x + screen_w;
			next_page = current_page;
			break;
		    case 's':
			if (!arg) {
			    int shrink = shrink_factor;
			    long fac1, fac2;
			    shrink_factor = 1;
			    fac1 = ROUNDUP(PAPER_WIDTH, screen_w);
			    fac2 = ROUNDUP(PAPER_HEIGHT, screen_h);
			    if (fac1 < fac2)
				number = fac2;
			    else
				number = fac1;
			    shrink_factor = shrink;
			}
			if (number <= 0) goto bad;
			if (number != shrink_factor) {
			    shrink_factor = number;
			    min_x = 0;
			    min_y = 0;
			    max_x = screen_w;
			    max_y = screen_h;
			    init_page();
			    define_conv();
			    reset_fonts();
			}
		    case '\f':
			/* redisplay current page */
			next_page = current_page;
			break;
		    case 'S':
			if (!arg) goto bad;
			if (number < 0) goto bad;
			if (number != density_factor) {
			  density_factor = number;
			  init_page();
			  define_conv();
			  reset_fonts();
			}
			/* redisplay current page */
			next_page = current_page;
			break;
		    case '\r':
		    case '\n':
			/* go to relative page */
			min_x = 0;
			min_y = 0;
			base_x = base_y = 0;
			max_x = screen_w;
			max_y = screen_h;
			next_page = current_page + (arg ? number : 1);
			break;
		    case 'g':
			/* go to absolute page */
			min_x = 0;
			min_y = 0;
			base_x = base_y = 0;
			max_x = screen_w;
			max_y = screen_h;
#ifdef POSTPS
			next_page = (arg ? (number?(number+POSTPSIND):1)
				     : total_pages) - 1;
#else /* POSTPS */
			next_page = (arg ? number : total_pages) - 1;
#endif /* POSTPS */
			break;
		    case '0': case '1': case '2': case '3': case '4':
		    case '5': case '6': case '7': case '8': case '9':
			if (! arg) {
				arg = 1;
				sign = 1;
				number = 0;
			}
			number = 10*number + sign*(ch - '0');
			continue;
		    case '-':
			if (! arg) {
				arg = 1;
				sign = -1;
				number = 0;
				continue;
			} else
				goto bad;
		    default:
			goto bad;
		}
		if (0 <= next_page && next_page < total_pages) {
			current_page = next_page;
			fseek(dvi_file, page_offset[current_page], 0);
			break;
		}
	bad:
#ifndef X10
		XBell(DISP, 10);
#else X10
		XFeep(0);
#endif X10
		arg = 0;		/* throw away numeric argument */
		continue;
	}
}

special(nbytes)
	unsigned long nbytes;
{
	char *cmd;
	int i;

	cmd = malloc((unsigned) nbytes+1);
	if (cmd == NULL)
		error("xdvi: Can't allocate memory for special (%d bytes)", nbytes);
	for (i = 0; i < nbytes; i += 1)
		cmd[i] = getc(dvi_file);
	cmd[i] = '\0';
	fprintf(stderr, "special ``%s'' not implemented\n", cmd);
	free(cmd);
}

/*
**
**      Read size bytes from the FILE fp, constructing them into a
**      signed/unsigned integer.
**
*/
unsigned long
num(fp, size)
	register FILE *fp;
	register int size;
{
        register int i;
	register long x;

	x = 0;
	for (i = 0; i < size; i += 1)
		x = x * 0x100 + (unsigned) getc(fp);
	return (x);
}

long
snum(fp, size)
	register FILE *fp;
	register int size;
{
        register int i;
	register long x;

	x = getc(fp) & 0xff;
	if (x & 0x80)
        	x -= 0x100;
	for (i = 1; i < size; i += 1)
	    x = x * 0x100 + (unsigned) getc(fp);
	return (x);
}

stop_output(sig)
{
	exit(sig);
}

/* VARARGS1 */
error(message, a, b, c, d, e, f)
	char *message;
{
	fprintf(stderr, message, a, b, c, d, e, f);
	putc('\n', stderr);
	exit(1);
}

init_page()
{
	page_h = PAPER_HEIGHT;
	page_w = PAPER_WIDTH;
}

clear_page()
{
#ifndef X10
	XClearWindow(DISP, win);
#else X10
	XClear(win);
#endif X10
}

put_border(x, y, w, h, t)
	long x, y, w, h, t;
{
	put_rectangle(x, y, w, t, forepix);
	put_rectangle(x, y, t, h, forepix);
	put_rectangle(x, y + h - t, w, t, forepix);
	put_rectangle(x + w - t, y, t, h, forepix);
}

put_rectangle(x, y, w, h, pix)
	long x, y, w, h;
#ifndef X10
	GC pix;
#else X10
	int pix;
#endif X10
{
#ifndef X10

#endif X10
	if (x < max_x && x + w >= min_x && y < max_y && y + h >= min_y)
#ifndef X10
		XFillRectangle(DISP, win, pix,
			       x - base_x, y - base_y, w?w:1, h?h:1);
#else X10
		XPixSet(win, x - base_x, y - base_y, w?w:1, h?h:1, pix);
#endif X10
}

put_bitmap(bitmap, x, y, pix)
	register struct bitmap *bitmap;
	register long x, y;
#ifndef X10
	GC pix;
#else X10
	int pix;
#endif X10
{
#ifndef X10
	XImage *image;

	if(!exposed && !(debug & DBG_BATCH)) return;
	image = XCreateImage(DISP, DefaultVisual(DISP, SCRN),
			     1, XYBitmap,
			     0, (char *)bitmap->bits,
			     bitmap->w, bitmap->h,
			     16, 0);
	image->bitmap_bit_order = LSBFirst;
	image->byte_order = LSBFirst;
#endif X10
        if (debug & DBG_BITMAP) printf("X(%d,%d)\n",x-base_x,y-base_y);
	if (x < max_x && x + bitmap->w >= min_x &&
	    y < max_y && y + bitmap->h >= min_y)
#ifndef X10
		XPutImage(DISP, win, pix, image,
			  0, 0,
			  x - base_x, y - base_y,
			  bitmap->w, bitmap->h);

#else X10
		XBitmapBitsPut(win, x - base_x, y - base_y,
				bitmap->w, bitmap->h, (char *) bitmap->bits,
				pix, backpix, NULL, GXfunc, AllPlanes);
#endif X10
}

sample(bitmap, x, y, w, h)
	register struct bitmap *bitmap;
	int x, y, w, h;
{
	register char *ptr, *endp;
	register int b, i, j, m, n;

	ptr = bitmap->bits
		+ (y * bitmap->bytes_wide)
		+ (x / BITS_PER_BYTE);
	endp = bitmap->bits + (bitmap->h * bitmap->bytes_wide);
	b = (1 << (x % BITS_PER_BYTE));
	n = 0;
	for (i = 0; i < h && ptr < endp; i += 1, ptr += bitmap->bytes_wide)
		for (m = b, j = 0; j < w; j += 1, m <<= 1)
			if (*ptr & m)
				n += 1;
	return ((n>0) &&
		density_factor?(n >= (i * w) / density_factor):n); /* was 3 */
}

shrink_bitmap(bitmap, x_factor, y_factor)
	register struct bitmap *bitmap;
	int x_factor, y_factor;
{
	char *shrunk_bits;
	int shrunk_height, shrunk_width, shrunk_bytes_wide;
	register char *ptr;
	char *cp;
	register int x, y, b, m;

	shrunk_height = ROUNDUP(bitmap->h, y_factor);
	shrunk_width = ROUNDUP(bitmap->w, x_factor);
	shrunk_bytes_wide = ROUNDUP(shrunk_width, BITS_PER_SHORT)*BYTES_PER_SHORT;
		/* width must be multiple of 16 bits for raster_op */
	ptr = shrunk_bits = calloc((unsigned) shrunk_height * shrunk_bytes_wide, 1);
	if (ptr == NULL)
		error("Can't allocate shrunken bitmap (%d by %d)",
			shrunk_height, shrunk_width);
	for (y = 0; y < bitmap->h; y += y_factor) {
		b = 0;
		m = (1 << 0);
		cp = ptr;
		for (x = 0; x < bitmap->w; x += x_factor) {
			if (sample(bitmap, x, y, x_factor, y_factor))
				*ptr |= m;
			else
				*ptr &= ~m;
			b += 1;
			m <<= 1;
			if (b % BITS_PER_BYTE == 0) {
				b = 0;
				m = (1 << 0);
				ptr += 1;
			}
		}
		ptr = cp + shrunk_bytes_wide;
	}
	free(bitmap->bits);
	bitmap->bits = shrunk_bits;
	bitmap->h = shrunk_height;
	bitmap->w = shrunk_width;
	bitmap->bytes_wide = shrunk_bytes_wide;
}

print_char(ch, g)
	ubyte ch;
	struct glyph *g;
{
	printf("char %d", ch);
	if (isprint(ch))
		printf(" (%c)", ch);
	putchar('\n');
	printf("x = %d, y = %d, pxl = %d, dvi = %d\n",
		g->x, g->y, g->pxl_adv, g->dvi_adv);
	print_bitmap(&g->bitmap);
}

print_bitmap(bitmap)
	register struct bitmap *bitmap;
{
	register char *ptr;
	register int x, y, i;

	ptr = bitmap->bits;
	if (ptr == NULL)
		return;
	printf("w = %d, h = %d, bytes wide = %d\n",
		bitmap->w, bitmap->h, bitmap->bytes_wide);
	for (y = 0; y < bitmap->h; y += 1) {
		for (x = 0; x < bitmap->bytes_wide; x += 1) {
			for (i = 0; i < BITS_PER_BYTE; i += 1)
				if (*ptr & (1 << i))
					putchar('@');
				else
					putchar(' ');
			ptr += 1;
		}
		putchar('\n');
	}
}

print_dvi(ch)
	ubyte ch;
{
	printf("%4d %4d ", PXL_H, PXL_V);
	if (ch <= SETCHAR0 + 127) {
		printf("SETCHAR%-3d", ch - SETCHAR0);
		if (isprint(ch))
			printf(" (%c)", ch);
	} else if (FNTNUM0 <= ch  &&  ch <= FNTNUM0 + 63) {
		printf("FNTNUM%d", ch - FNTNUM0);
	} else {
		switch (ch) {
		    case SET1:
			printf("SET1");
			break;
		    case SETRULE:
			printf("SETRULE");
			break;
		    case PUT1:
			printf("PUT1");
			break;
		    case PUTRULE:
			printf("PUTRULE");
			break;
		    case NOP:
			printf("NOP");
			break;
		    case BOP:
			printf("BOP");
			break;
		    case EOP:
			printf("EOP");
			break;
		    case PUSH:
			printf("PUSH");
			break;
		    case POP:
			printf("POP");
			break;
		    case RIGHT1:
		    case RIGHT2:
		    case RIGHT3:
		    case RIGHT4:
			printf("RIGHT%d", ch - RIGHT1 + 1);
			break;
		    case X0:
		    case X1:
		    case X2:
		    case X3:
		    case X4:
			printf("X%d", ch - X0);
			break;
		    case W0:
		    case W1:
		    case W2:
		    case W3:
		    case W4:
			printf("W%d", ch - W0);
			break;
		    case Y0:
		    case Y1:
		    case Y2:
		    case Y3:
		    case Y4:
			printf("Y%d", ch - Y0);
			break;
		    case Z0:
		    case Z1:
		    case Z2:
		    case Z3:
		    case Z4:
			printf("Z%d", ch - Z0);
			break;
		    case DOWN1:
		    case DOWN2:
		    case DOWN3:
		    case DOWN4:
			printf("DOWN%d", ch - DOWN1 + 1);
			break;
		    case FNT1:
		    case FNT2:
		    case FNT3:
		    case FNT4:
			printf("FNT%d", ch - FNT1 + 1);
			break;
		    case XXX1:
		    case XXX2:
		    case XXX3:
		    case XXX4:
			printf("XXX%d", ch - XXX1 + 1);
			break;
		    case FNTDEF1:
		    case FNTDEF2:
		    case FNTDEF3:
		    case FNTDEF4:
			printf("FNTDEF%d", ch - FNTDEF1 + 1);
			break;
		    case PRE:
			printf("PRE");
			break;
		    case POST:
			printf("POST");
			break;
		    case POSTPOST:
			printf("POSTPOST");
			break;
		    default:
			error("xdvi: Unknown op-code %d, offset %d",
				ch, ftell(dvi_file));
		} /* end switch*/
	} /* end else (ch not a SETCHAR or FNTNUM) */
	putchar('\n');
}
