/* $Header: obj_block.c,v 1.3 88/08/31 23:43:44 moraes Exp $ */
/* 
 *  The block edit routines - treated as a separate pseudo-object
 *  because they are so similar. The code is somewhat intricate - the
 *  result of trying to cram as many operations into as little code as
 *  possible, so that the code to do something stayed in the one place.
 *  Sigh! macros might have been nicer.
 */
#include <values.h>

#include "xpic.h"
#include "windows.h"
#include "spline.h"
#include "gels.h"
#include "draw.h"
#include "input.h"
#include "newfonts.h"
#include "assert.h"

static int x1, y1, x2, y2;			/* Corners of box, ellipse, ends of line */
static int xmin, xmax, ymin, ymax;	/* Bounding box */
static Gel *gel;
static Gel *oldgel;
static Cell *cell;
static Box *bp;
static Box adjbox;
static int lastX, lastY;
static int first_time = FALSE;

/*
 *  For the editing constructs, we also may have a DRAG_MODE (where the
 *  object is dragged around, or an ASK_MODE where the user is asked to
 *  confirm the operation (usually a delete, with a click
 */
block_event(evtype, mx, my)
{
	char *err;
	register Gel *tmp;
	
	switch(evtype) {
	case MOTION | START_MODE:
	case MOTION | ASK_MODE:
	case RIGHT  | START_MODE:
	case MIDDLE | START_MODE:
	case REDRAW | START_MODE:
	case REDRAW | ASK_MODE:
		break;
	case MOTION | END_MODE:
		/* rubber band the box corner */
		box(picWin, x1, y1, x2, y2, gcBlock);
		x2 = mx;
		y2 = my;
		box(picWin, x1, y1, x2, y2, gcBlock);
		break;
	case MOTION | DRAG_MODE:  
		/*
		 *  move the box around on the cursor - use the second corner as
		 *  the mouse corner
		 */
		box(picWin, x1, y1, x2, y2, gcBlock);
		x1 += mx - x2;
		y1 += my - y2;
		x2 = mx;
		y2 = my;
		box(picWin, x1, y1, x2, y2, gcBlock);
		break;
	case LEFT | START_MODE:
		/* start the box */
		first_time = TRUE;
		if (editType == PASTE) {
			if (KillBuffer == NULL) {
				message("Nothing to paste. Cut something first");
				break;
			}
			gel = CopyGel(KillBuffer, MAXINT);
			bp = GetBBox(gel);
			x1 = bp->ll.x;
			y1 = bp->ll.y;
			lastX = x2 = bp->ur.x;
			lastY = y2 = bp->ur.y;
			drawingMode = DRAG_MODE;
		} else if (editType == GET) {
			if (cell)
				FreeCell(cell);
			cell = ReadCell("Get from file ? ", NULL);
			if ( cell == NULL) 
				break;
			else if (cell->gelList == NULL) {
				message("Can't paste an empty cell!");
				FreeCell(cell);
				cell = NULL;
				break;
			}
			gel = cell->gelList;
			cell->gelList = NULL;
			bp = GetBBox(gel);
			x1 = bp->ll.x;
			y1 = bp->ll.y;
			lastX = x2 = bp->ur.x;
			lastY = y2 = bp->ur.y;
			drawingMode = DRAG_MODE;
		} else if (editType == SCALE || editType == ROTATE) {
			message("SCALE/ROTATE not yet implemented");
			break;
		} else {
			x1 = x2 = mx;
			y1 = y2 = my;
			drawingMode = END_MODE;
		}
		box(picWin, x1, y1, x2, y2, gcBlock);
		break;
	case LEFT | END_MODE:
		/* Won't get here for GET, PASTE - they go straight to DRAG */
		/* End the box, find the contained gels, highlight them */
		box(picWin, x1, y1, x2, y2, gcBlock);
		lastX = x2 = mx;
		lastY = y2 = my;
		xmin = MIN(x1, mx);
		xmax = MAX(x1, mx);
		ymin = MIN(y1, my);
		ymax = MAX(y1, my);
		/*
		 *  The very nature of ADJUST means that we want
		 *  intersecting Gels - for the others, we want only
		 *  those that are strictly contained within the box
		 */
		if (editType == ADJUST) {
			oldgel = FindIntersectingGels(&(CurrentCell->gelList), 
			 xmin, ymin, xmax, ymax);
			gel = CopyGel(oldgel, MAXINT);
			adjbox.ll.x = xmin;
			adjbox.ll.y = ymin;
			adjbox.ur.x = xmax;
			adjbox.ur.y = ymax;
			err = "No intersecting elements";
		} else {
			oldgel = FindContainedGels(&(CurrentCell->gelList), 
			 xmin, ymin, xmax, ymax);
			gel = CopyGel(oldgel, MAXINT);
			err = "No contained elements";
		}
		if (oldgel == NULL) {
			message(err);
			drawingMode = START_MODE;
			break;
		}
		if (editType == DELETE || editType == CHANGE_ATTRIBUTE) {
			GelHilite(gel);
			drawingMode = ASK_MODE;
			sprintf(errstring, "Click Left button to confirm %s",
			 (editType == DELETE) ? "Delete" : "Change");
			message(errstring);
		} else if (editType == COPY || editType == MOVE || 
		 editType  == ADJUST) {
		 	if (editType == COPY) {
				/* 'gel' must be a copy of the gels selected */
				PushUnderUndo(&(CurrentCell->gelList), oldgel, 
				 CurrentCell->undo);
				oldgel = NULL;
			}
			GelHilite(gel);
			drawingMode = DRAG_MODE;
			/* Draw the rubber banded box for drag */
			box(picWin, x1, y1, x2, y2, gcBlock);
		} else if (editType == PUT) {
			if (cell)
				FreeCell(cell);
			PushGel(&(CurrentCell->gelList), oldgel);
			GelHilite(gel);
			if ((cell = NewCell(NULL, nullfile)) != NULL) {
				cell->gelList = gel;
				cell->saved |= MODIFIED;
				/* 
				 *  If we try to write out a cell with a name of
				 *  nullfile, it will ask for the name
				 */
				WriteCell(cell, backupOnWrite);
				cell->gelList = NULL;
			}
			GelUnHilite(gel);
			FreeGel(gel);
			gel = NULL;
			drawingMode = START_MODE;
		} else { /* shouldn't get here, since SCALE, ROTATE aren't working! */ 
			sprintf(errstring,  "Unknown editType %d in block_event - LEFT", 
			 editType);
			message(errstring);
			drawingMode = START_MODE;
		}
		break;
	case LEFT | DRAG_MODE:
		/* 
		 *  Every time LEFT is clicked in DRAG mode, we move all
		 *  the objects in the gel list here, highlight them,
		 *  and stay in this mode - gives the users multiple
		 *  tries at adjusting the block since it isn't
		 *  completely WYSIWYG. PUT doesn't get here since it
		 *  ends in END_MODE, DELETE and CHANGE_ATTRIB go via
		 *  ASK_MODE
		 */
		box(picWin, x1, y1, x2, y2, gcBlock);
		x1 += mx - x2;
		y1 += my - y2;
		x2 = mx;
		y2 = my;
		if (first_time) {
			first_time = FALSE;
		 	if (editType == COPY) {
				GelUnHilite(gel);
			} else if (editType == MOVE || editType == ADJUST) {
				GelUnHilite(gel);
				for (tmp = gel; tmp != NULL; tmp = tmp->next) 
					GelDraw(tmp, ERASE);
			} else { /* GET, PASTE */
				/* Do nothing */
			}
		} else { /* Not first time, so we zonk the last position */
			GelUnHilite(gel);
			for (tmp = gel; tmp != NULL; tmp = tmp->next) 
				GelDraw(tmp, INVERT);
		}
		if (editType == ADJUST) {
			AdjustGel(gel, &adjbox, mx - lastX, my - lastY);
			adjbox.ll.x = MIN(x1, x2);
			adjbox.ll.y = MIN(y1, y2);
			adjbox.ur.x = MAX(x1, x2);
			adjbox.ur.y = MAX(y1, y2);
		} else
			MoveGel(gel, mx - lastX, my - lastY);
		lastX = mx;
		lastY = my;
		for (tmp = gel; tmp != NULL; tmp = tmp->next) 
			GelDraw(tmp, INVERT);
		GelHilite(gel);
		box(picWin, x1, y1, x2, y2, gcBlock);
		break;
	case RIGHT | DRAG_MODE:
		box(picWin, x1, y1, x2, y2, gcBlock);
		if ((!first_time) || editType == COPY || editType == MOVE || 
		 editType == ADJUST) {
			GelUnHilite(gel);
			if (!first_time)
				for (tmp = gel; tmp != NULL; tmp = tmp->next)
					GelDraw(tmp, INVERT);
		}
		if (!(first_time && (editType == GET || editType == PASTE))) {
			for (tmp = gel; tmp != NULL; tmp = tmp->next) 
				GelDraw(tmp, DRAW);
			FreeGel(CurrentCell->undoList);
			CurrentCell->undoList = NULL;
			if (editType != GET && editType != PASTE) {
				CurrentCell->undoList = oldgel;
				oldgel = NULL;
			}
			CurrentCell->undo = PushGel(&(CurrentCell->gelList), 
			 gel);
			CurrentCell->saved |= MODIFIED;
		}
		gel = NULL;
		drawingMode = START_MODE;
		break;		
	case LEFT  | ASK_MODE:
		/* Delete, change the gels */
		GelUnHilite(gel);
		for (tmp = gel; tmp != NULL; tmp = tmp->next) 
			GelDraw(tmp, ERASE);
		FreeGel(CurrentCell->undoList);
		CurrentCell->undoList = oldgel;
		oldgel = NULL;
		if (editType == DELETE) {
			CurrentCell->undo = 0;
			FreeGel(KillBuffer);
			KillBuffer = gel;
		} else { /* CHANGE_ATTRIB */
			ChangeAttrib(gel, line_type, line_arrow, lineThickness, 
			 fill_type, 
			 textVertAlign | textHorizAlign, fontType, textSize);
			for (tmp = gel; tmp != NULL; tmp = tmp->next) 
				GelDraw(tmp, DRAW);
			CurrentCell->undo = PushGel(&(CurrentCell->gelList), gel);
		}
		gel = NULL;
		CurrentCell->saved |= MODIFIED;
		drawingMode = START_MODE;
		break;
	case RIGHT | ASK_MODE:
	case MIDDLE | ASK_MODE:
		/* Abort */
		GelUnHilite(gel);
		PushUnderUndo(&(CurrentCell->gelList), gel, CurrentCell->undo);
		gel = NULL;
		drawingMode = START_MODE;
		break;
	case MIDDLE | DRAG_MODE:
		/* Abort, unhilite the gels - delete any copies */
		box(picWin, x1, y1, x2, y2, gcBlock);
		/* PUT only gets as far as END_MODE - won't get here */
		if ((!first_time) || editType == COPY || editType == MOVE || 
		 editType == ADJUST) {
			GelUnHilite(gel);
			if (!first_time)
				for (tmp = gel; tmp != NULL; tmp = tmp->next)
					GelDraw(tmp, INVERT);
		}
		if (editType == COPY || editType == PASTE || editType == GET) {
			FreeGel(gel);
			if (cell)
				FreeCell(cell);
			gel = NULL;
			cell = NULL;
		} else if (editType == MOVE || editType == ADJUST) {
			FreeGel(gel);
			gel = oldgel;
			for (tmp = gel; tmp != NULL; tmp = tmp->next) 
				GelDraw(tmp, DRAW);
			PushUnderUndo(&(CurrentCell->gelList), gel, CurrentCell->undo);
			gel = NULL;
		} else { /* can't happen, since we don't have SCALE/ROTATE */
			sprintf(errstring, "Unknown editType %d in block_event - MIDDLE", 
			 editType);
			message(errstring);
		}
		drawingMode = START_MODE;
		break;
	case RIGHT | END_MODE:
	case MIDDLE | END_MODE:
		/* Abort - stop rubber banding */
		box(picWin, x1, y1, x2, y2, gcBlock);
		drawingMode = START_MODE;
		break;
	case REDRAW | END_MODE:
		/* redraw the rubber band */
		box(picWin, x1, y1, x2, y2, gcBlock);
		break;
	case REDRAW | DRAG_MODE:
		/* highlight the gels, redraw the drag box */
		box(picWin, x1, y1, x2, y2, gcBlock);
		if (!first_time || (editType == COPY || editType == MOVE 
		 || editType == ADJUST)) 
			GelHilite(gel);
		break;
	default:
#ifdef DEBUG
		sprintf(errstring, "Hey! Unknown BLOCK mode %d", drawingMode);
		message(errstring);
#endif
		break;
	}
	ASSERT(allock(), "block_event");
}

block_abort()
{
	/* Fudge up a RIGHT button pressed event - safer thing to do */
	block_event((RIGHT | drawingMode), 0, 0);
}
	
