/* 
 * qmhide.c - Do hidden surface elimination for a set of quadmeshes.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <stdio.h>
#include "X.h"
#include "pixmap.h"
#include "gc.h"
#include "miscstruct.h"
#include "gcstruct.h"
#include "extnsionst.h"
#include "dix.h"
#include "dixstruct.h"
#include "resource.h"
#include "colormap.h"

#include "PEX.h"
#include "pubstr.h"
#include "PEXproto.h"
#include "quadmesh.h"
#include "renderer.h"
#include "pc.h"
#include "qmhide.h"
#include "colortable.h"
#include "mipex.h"

/* Structs & defines used by seed fill in make_region */
struct seg {
    short y, xl, xr, dy;	/* horizontal segment of scanline y */
};

#define MAX_STACK 1000		/* max depth of stack */

#define PUSH(Y, XL, XR, DY) \
    if (sp<stack+MAX_STACK && Y+(DY)>=0 && Y+(DY)<pQA->n) \
    {sp->y = Y; sp->xl = XL; sp->xr = XR; sp->dy = DY; sp++;}

#define POP(Y, XL, XR, DY) \
    {sp--; Y = sp->y+(DY = sp->dy); XL = sp->xl; XR = sp->xr;}

/* Debugging */
static int dbg_depth = -1;

/*****************************************************************
 * TAG( qm_insert )
 * 
 * Create a qm_facet_array corresponding to the given quad mesh and
 * add to the list.
 * Inputs:
 * 	pQM:	Pointer to the quad mesh.
 * 	ppQA:	Pointer to pointer to head of qm_facet_arrays.
 * Outputs:
 * 	ppQA:	Modified by adding newly created qm_facet_array.
 * Assumptions:
 * 	
 * Algorithm:
 * 	Creates a qm_facet_array the same size as the PEXQuadMesh, and
 * 	initializes all the fields from the PEXQuadMesh.
 */
void
qm_insert( pQM, ppQA, pRD )
pexQuadMeshPtr pQM;
qm_facet_array **ppQA;
pexRendererPtr *pRD;
{
    qm_facet_array * pQA;
    qm_facet * pQF, * pRow;
    int i, j, ii, jj;
    Bool use_w;
    pexCoord3D * pc, pt, corners[2][2], diag[2], nrml;
    float w;

    if ( dbg_depth >= 0 )
	printf( "qm_insert( pQM=0x%x, ppQA=0x%x )\n", pQM, ppQA );
    use_w = pQM->pointType == Rational;

    pQA = (qm_facet_array *)Xalloc(
	sizeof(qm_facet_array) +
	sizeof(qm_facet) * (pQM->mPoints - 1) * (pQM->nPoints - 1) );
    pQA->n = pQM->nPoints - 1;
    pQA->m = pQM->mPoints - 1;
    pQA->n_stride = sizeof(qm_facet);
    pQA->m_stride = pQA->n_stride * pQA->n;
    pQA->offset = sizeof(qm_facet_array);
    pQA->mesh = pQM;
    pQA->color = pRD->pPC->surfaceColor;

    for ( i = 0, pRow = qmFirst(pQA); i < pQA->m;
	  i++, pRow = qmNextM(pQA, pRow) )
	for ( j = 0, pQF = pRow; j < pQA->n; j++, pQF = qmNextN(pQA, pQF ) )
	{
	    pQF->arr = pQA;
	    for ( ii = 0; ii < 2; ii++ )
		for ( jj = 0; jj < 2; jj++ )
		{
		    pc = PexQuadMeshVertex(pQM, i + ii, j + jj);
		    if ( use_w )
		    {
			w = ((pexCoord4D *)pc)->w;
			if ( w == 0 ) w = 1; 
			pt.x = pc->x / w;
			pt.y = pc->y / w;
			pt.z = pc->z / w;
			pc = &pt;
		    }

		    corners[ii][jj] = *pc;
		    if ( ii == 0 && jj == 0 )
		    {
			pQF->midpoint_z = pc->z;
			pQF->bbox[0][0] = pQF->bbox[0][1] = pc->x;
			pQF->bbox[1][0] = pQF->bbox[1][1] = pc->y;
			pQF->bbox[2][0] = pQF->bbox[2][1] = pc->z;
		    }
		    else
		    {
			pQF->midpoint_z += pc->z;
			if ( pc->x < pQF->bbox[0][0] )
			    pQF->bbox[0][0] = pc->x;
			if ( pc->x > pQF->bbox[0][1] )
			    pQF->bbox[0][1] = pc->x;
			if ( pc->y < pQF->bbox[1][0] )
			    pQF->bbox[1][0] = pc->y;
			if ( pc->y > pQF->bbox[1][1] )
			    pQF->bbox[1][1] = pc->y;
			if ( pc->z < pQF->bbox[2][0] )
			    pQF->bbox[2][0] = pc->z;
			if ( pc->z > pQF->bbox[2][1] )
			    pQF->bbox[2][1] = pc->z;
		    }
		}

	    pQF->midpoint_z /= 4;

	    diag[0].x = corners[1][1].x - corners[0][0].x;
	    diag[0].y = corners[1][1].y - corners[0][0].y;
	    diag[0].z = corners[1][1].z - corners[0][0].z;
	    diag[1].x = corners[0][1].x - corners[1][0].x;
	    diag[1].y = corners[0][1].y - corners[1][0].y;
	    diag[1].z = corners[0][1].z - corners[1][0].z;
	    nrml.x = diag[0].y * diag[1].z - diag[0].z * diag[1].y;
	    nrml.y = diag[0].z * diag[1].x - diag[0].x * diag[1].z;
	    nrml.z = diag[0].x * diag[1].y - diag[0].y * diag[1].x;
	    pQF->pln_eq[0] = nrml.x;
	    pQF->pln_eq[1] = nrml.y;
	    pQF->pln_eq[2] = nrml.z;
	    pQF->pln_eq[3] = -(nrml.x * corners[0][0].x +
			       nrml.y * corners[0][0].y +
			       nrml.z * corners[0][0].z);
	    /* Get variance */
	    pQF->pln_min = pQF->pln_max = 0.0;
	    for ( ii = 0; ii < 2; ii++ )
		for ( jj == 0; jj < 2; jj++ )
		{
		    w = (pQF->pln_eq[0]*corners[ii][jj].x +
			 pQF->pln_eq[1]*corners[ii][jj].y +
			 pQF->pln_eq[2]*corners[ii][jj].z +
			 pQF->pln_eq[3]);
		    if ( w < pQF->pln_min )
			pQF->pln_min = w;
		    if ( w > pQF->pln_max )
			pQF->pln_max = w;
		}
	    pQF->i = i;
	    pQF->j = j;
	    pQF->n_behind = 0;
	    pQF->region = 0;
	    pQF->rendered = 0;

	    pQF->visible = ins_sqr_facet( pQF );
	}
    

    /* Add to list */
    pQA->next = *ppQA;
    *ppQA = pQA;
}

#if 0				/* not used */
/*****************************************************************
 * TAG( make_region )
 * 
 * Make a new region from adjacent facets with same direction normals.
 * Inputs:
 * 	pQF:	Pointer to "seed" facet.
 * Outputs:
 * 	Sets region field in all facets 4-connected to seed facet,
 * 	which have  normal pointing in same Z dir as seed facet.
 * Assumptions:
 *	[None]
 * Algorithm:
 * 	Modified seed fill algorithm from Paul Heckbert.
 */
static void
make_region( pQF )
qm_facet * pQF;
{
    static int rgn = 0;
    int x, y;
    int l, x1, x2, dy;
    int dir;			/* normal direction */
    struct seg stack[MAX_STACK], *sp = stack;	/* stack of filled segments */
    qm_facet_array * pQA;
    int dbg_ctr;

    ++rgn;

    if ( dbg_depth >= 0 )
	printf( "make_region( 0x%x[%d,%d] ) -> %d\n", pQF, pQF->i, pQF->j, rgn );

    if ( pQF->pln_eq[2] == 0.0 )
    {
	pQF->region = rgn;
	return;
    }
    dir = pQF->pln_eq[2] > 0 ? 1 : -1;
    x = pQF->i;
    y = pQF->j;
    pQA = pQF->arr;

    PUSH(y, x, x, 1);			/* needed in some cases */
    PUSH(y+1, x, x, -1);		/* seed segment (popped 1st) */

    dbg_ctr = 0;

    while (sp>stack) {
	/* pop segment off stack and fill a neighboring scan line */
	POP(y, x1, x2, dy);
	/*
	 * segment of scan line y-dy for x1<=x<=x2 was previously filled,
	 * now explore adjacent quads in scan line y
	 */
	for (x=x1;
	     x>=0 && (pQF=qmFacet(pQA,x,y))->pln_eq[2]*dir > 0 &&
	     pQF->region != rgn;
	     x--)
	{
		pQF->region = rgn;
	    if ( dbg_depth >= -1 )
	    {
		if ( (++dbg_ctr % 6) == 0 )
		    printf( "\n" );
		printf( "\t[%d,%d]", x, y );
	    }
	}
	    
	if (x>=x1) goto skip;
	l = x+1;
	if (l<x1) PUSH(y, l, x1-1, -dy);		/* leak on left? */
	x = x1+1;
	do {
	    for (; x<pQA->m && (pQF=qmFacet(pQA,x,y))->pln_eq[2]*dir > 0 &&
		 pQF->region != rgn; 
		 x++)
	    {
		pQF->region = rgn;
		if ( dbg_depth >= 0 )
		{
		    if ( (++dbg_ctr % 6) == 0 )
			printf( "\n" );
		    printf( "\t[%d,%d]", x, y );
		}
	    }

	    PUSH(y, l, x-1, dy);
	    if (x>x2+1) PUSH(y, x2+1, x-1, -dy);	/* leak on right? */
skip:	    for (x++; x<=x2 && (pQF=qmFacet(pQA,x,y))->pln_eq[2]*dir > 0; x++)
    		;
	    l = x;
	} while (x<=x2);
    }
    if ( dbg_depth >= 0 )
	printf( "\n" );
}
#endif


/*****************************************************************
 * TAG( xy_overlap )
 * 
 * Return TRUE if X/Y part of bounding boxes of facets overlap.
 * Inputs:
 * 	pQF1, pQF2:	Pointers to facets.
 * Outputs:
 * 	TRUE if bounding boxes overlap, FALSE otherwise.
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
static int
xy_overlap( pQF1, pQF2 )
register qm_facet * pQF1, * pQF2;
{
    return !(pQF1->bbox[0][0] > pQF2->bbox[0][1] ||
	     pQF1->bbox[0][1] < pQF2->bbox[0][0] ||
	     pQF1->bbox[1][0] > pQF2->bbox[1][1] ||
	     pQF1->bbox[1][1] < pQF2->bbox[1][0]);
}


#define FRONT	1
#define BACK	2
#define OVERLAP	3

#if 0
/*****************************************************************
 * TAG( check_plane )
 * 
 * Check the first facet against the plane of the second.
 * Inputs:
 * 	pQF1:		First facet to check.
 * 	pQF2:		Second facet - check against its plane.
 * Outputs:
 * 	Returns FRONT if the facet is in front of the plane (larger Z)
 * 	BACK if the facet is in back of the plane, and
 * 	OVERLAP if neither.
 * Assumptions:
 *	[None]
 * Algorithm:
 * 	Substitutes the facet vertices into the plane equation.  If
 * 	all are < 0 (and normal points towards positive Z), then BACK.
 * 	If all are > 0, then FRONT.  Otherwise OVERLAP.  Ignore
 * 	vertices that are "too close" to the plane.
 */
static int
check_plane( pQF1, pQF2 )
register qm_facet * pQF1, * pQF2;
{
    pexCoord3D *pt;
    int retval = OVERLAP;
    float val;
    register pexQuadMeshPtr pQM;
    int i, j;
    Bool use_w;

    if ( dbg_depth >= 0 )
	printf( "%.*scheck_plane( [%d,%d], pln[%d,%d] ) -> ",
		dbg_depth,
		"                                                            ",
		pQF1->i, pQF1->j, pQF2->i, pQF2->j );

    pQM = pQF1->arr->mesh;
    use_w = pQM->pointType == Rational;

    for ( i = 0; i < 2; i++ )
	for ( j = 0; j < 2; j++ )
	{
	    pt = PexQuadMeshVertex(pQM, pQF1->i + i, pQF1->j + j);
	    if ( use_w )
		val = pQF2->pln_eq[0]*pt->x + pQF2->pln_eq[1]*pt->y +
		    pQF2->pln_eq[2]*pt->z +
			pQF2->pln_eq[3] * ((pexCoord4D *)pt)->w;
	    else
		val = pQF2->pln_eq[0]*pt->x + pQF2->pln_eq[1]*pt->y +
		    pQF2->pln_eq[2]*pt->z + pQF2->pln_eq[3];
	    if ( pQF2->pln_eq[2] > 0 )
		val = -val;
	    if ( val < pQF2->pln_min )
	    {
		if ( retval == FRONT )
		{
		    if ( dbg_depth >= 0 ) printf( "OVERLAP\n" );
		    return OVERLAP;
		}
		else
		    retval = BACK;
	    }
	    else if ( val > pQF2->pln_max )
	    {
		if ( retval == BACK )
		{
		    if ( dbg_depth >= 0 ) printf( "OVERLAP\n" );
		    return OVERLAP;
		}
		else
		    retval = FRONT;
	    }
	}
    if ( dbg_depth >= 0 )
	printf( retval == FRONT ? "FRONT\n" :
		retval == BACK ? "BACK\n" : "OVERLAP\n" );
    return retval;
}
#endif

/*****************************************************************
 * TAG( facet_compare )
 * 
 * Compare two facets to determine whether one is behind the other.
 * Inputs:
 * 	pQF1, pQF2:	Facets to be compared.
 * Outputs:
 * 	Returns BACK if facet 2 is in back of facet 1, FRONT if facet
 * 	2 is in front of facet 1, 0 otherwise.
 * Assumptions:
 * 	Facets do not interpenetrate.
 * Algorithm:
 * 	If facets are in same region, return 0.
 *	If facet 2 is on facet 1's behind list, return BACK.
 * 	If bounding boxes don't overlap, return 0.
 * 	If facet 1.zmin < facet 2.zmax, return BACK.
 * 	If converse, return FRONT.
 * 	If facet1 is in front of facet 2 plane, or facet 2 is in back
 * 	of facet 1 plane, return BACK.
 * 	If converse, return FRONT.
 * 	Else punt, and return 0.
 */
static int
facet_compare( pQF1, pQF2 )
register qm_facet * pQF1, * pQF2;
{
    register int i;

    if ( pQF2->rendered )
	return BACK;
#ifdef notdef
    if ( pQF2->region == 0 )
	make_region( pQF2 );
    if ( pQF1->region == pQF2->region )
	return 0;
#endif
    /* Are they adjacent with similar normals? */
    if ( pQF1->arr == pQF2->arr &&
	 abs(pQF2->i - pQF1->i) + abs(pQF2->j - pQF1->j) <= 1 &&
	 pQF2->pln_eq[2] * pQF1->pln_eq[2] >= 0.0 )
	return 0;
    for ( i = 0; i < pQF1->n_behind; i++ )
	if ( pQF1->behind[i] == pQF2 )
	    return BACK;
    if ( ! xy_overlap( pQF1, pQF2 ) )
	return 0;
    if ( pQF1->bbox[2][0] >= pQF2->bbox[2][1] )
	return FRONT;
    if ( pQF1->bbox[2][1] <= pQF2->bbox[2][0] )
	return BACK;

#if 0
    switch( check_plane( pQF1, pQF2 ) )
    {
    case FRONT:			/* Facet 1 in front of facet 2 plane */
	return BACK;
    case BACK:
	/* If possible, add facet 1 to facet 2's behind list */
	if ( pQF2->n_behind < MAX_BEHIND )
	    pQF2->behind[pQF2->n_behind++] = pQF1;
	return FRONT;
    }
    switch( check_plane( pQF2, pQF1 ) )
    {
    case FRONT:
	/* If possible, add facet 1 to facet 2's behind list */
	if ( pQF2->n_behind < MAX_BEHIND )
	    pQF2->behind[pQF2->n_behind++] = pQF1;
	return FRONT;
    case BACK:
	return BACK;
    }
#endif
    /* Test midpoints */
    if ( pQF2->midpoint_z > pQF1->midpoint_z )
	return BACK;
    else
	return FRONT;
}

/*****************************************************************
 * TAG( facet_draw )
 * 
 * Draw a single facet using the DrawQuadMesh2D routines.
 * Inputs:
 * 	pQF:		Pointer to the facet to be drawn.
 * 	pRD:		Pointer to the renderer.
 * Outputs:
 * 	Calls either the smooth or flat drawpolygon2d routine to draw
 * 	the facet.
 * Assumptions:
 * 	shure.
 * Algorithm:
 * 	Build a little FillArea3DWithData and draw it.  Copy normals
 * 	as necessary.
 * 	*** Actually, for now, copy code from polygon.c.
 */
void
facet_draw( pQF, pRD )
qm_facet *pQF;
pexRendererPtr pRD;
{
    /* Call one of miPexQuadMesh2DFlatHide or miPexQuadMesh2DSmoothHide */
    if ( pQF->arr->mesh->vertexAttributes & GANormal )
	(*pRD->QuadMesh2DSmooth)(pRD, pQF);	/* may be either */
    else
	(*pRD->QuadMesh2DFlat)(pRD, pQF);	/* always flat */
}


/*****************************************************************
 * TAG( facet_render )
 * 
 * Render a facet, first rendering all facets behind it.
 * Inputs:
 * 	QAlist:		List of all quad meshes.
 * 	pQF:		Pointer to facet to be rendered.
 * 	pRD:		Pointer to renderer.
 * Outputs:
 * 	Renders facet and all facets behind it.
 * Assumptions:
 * 	you bet.
 * Algorithm:
 * 	Compare to all potentially overlapping facets.  Recursively
 * 	render all those that are behind.  Finally, draw the facet.
 */
static void
facet_render( QAlist, pQF, pRD )
qm_facet_array * QAlist;
qm_facet * pQF;
pexRendererPtr pRD;
{
    qm_facet * pQF2;
    int i, j, n;
    int minx, maxx, miny, maxy;

    if ( !pQF->visible || pQF->rendered )
	return;
    if ( dbg_depth >= 0 )
    {
	dbg_depth++;
	printf( "%.*sfacet_render( 0x%x[%d,%d] )\n", dbg_depth,
		"                                                            ",
		pQF, pQF->i, pQF->j );
    }
    pQF->rendered = 1;
#ifdef notdef
    if ( pQF->region == 0 )
	make_region( pQF );
#endif
    /* Compare against all facets in the same square */
    if ( ! facet_sqrs( pQF, &minx, &maxx, &miny, &maxy ) )
	return;			/* shouldn't happen */
    for ( j = miny; j <= maxy; j++ )
	for ( i = minx; i <= maxx; i++ )
	    for ( n = 0; ; n++ )
		if ( pQF2 = get_sqr_facet( i, j, n ) )
		{
		    if ( facet_compare( pQF, pQF2 ) == BACK )
			facet_render( QAlist, pQF2, pRD );
		}
		else
		    break;	/* N loop */

    facet_draw( pQF, pRD );
    if ( dbg_depth >= 0 )
	dbg_depth--;
}


/*****************************************************************
 * TAG( qmhide )
 * 
 * Do "painters algorithm" hidden surface elimination on the given
 * list of quad meshes.
 * Inputs:
 * 	QAlist:		List of quad meshes.
 * 	pRD:		Pointer to renderer.
 * Outputs:
 * 	Draws all facets from back to front.
 * Assumptions:
 *	[None]
 * Algorithm:
 * 	For each facet, draw it in front of all facets behind it.
 */
void
qmhide( QAlist, pRD )
qm_facet_array *QAlist;
pexRendererPtr pRD;
{
    qm_facet * pQF, * pQFrow;
    int i, j;

    for ( ; QAlist; QAlist = QAlist->next )
	for ( pQFrow = qmFirst(QAlist), i = 0;
	      i < QAlist->m;
	      i++, pQFrow = qmNextM(QAlist, pQFrow) )
	    for ( j = 0, pQF = pQFrow;
		  j < QAlist->n;
		  j++, pQF = qmNextN(QAlist, pQF) )
		facet_render( QAlist, pQF, pRD );
}


/*****************************************************************
 * TAG( miPexQuadMesh2DFlatHide )
 * 
 * 2D Drawing support function for quadmesh rendering.
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
miPexQuadMesh2DFlatHide(pRD, pQF)
pexRendererPtr pRD;
qm_facet * pQF;
{
    pexQuadMeshPtr qa = pQF->arr->mesh;
    pexCoord4D pts[4];		/* Input points */
    pexCoord4D cpt[12];		/* clipped points */
    DDXPointRec ddxpt[12];	/* GC coordinate endpoints */
    DrawablePtr pDraw;
    GCPtr pGC;
    Pixel index;
    pexVector3D normal;
    int shade, vis, i;


    if ( dbg_depth >= 0 )
	printf( "%.*sfacet_draw( 0x%x[%d,%d] )\n", dbg_depth,
		"                                                            ",
		pQF, pQF->i, pQF->j );

    pDraw = pRD -> pDraw;
    pGC =  pRD -> pGC;
    
    pts[0] = *(pexCoord4D *)PexQuadMeshVertex(qa, pQF->i, pQF->j);
    pts[1] = *(pexCoord4D *)PexQuadMeshVertex(qa, pQF->i+1, pQF->j);
    pts[2] = *(pexCoord4D *)PexQuadMeshVertex(qa, pQF->i+1, pQF->j+1);
    pts[3] = *(pexCoord4D *)PexQuadMeshVertex(qa, pQF->i, pQF->j+1);

    if ((pRD->pPC->cullMode == BackFaces) &&
	PexCullFacet (pts, 4))
	return Success;

    /* Clip the polygon */
    vis = PexClipPolygon(4, pts, 12, cpt);
    if ( vis )
    {
	for ( i = 0; i < vis; i++ )
	{
	    /* Perspective divide (finally!) */
	    if ( cpt[i].w != 1.0 )
	    {
		cpt[i].x /= cpt[i].w;
		cpt[i].y /= cpt[i].w;
	    }
	    /* Transform to DC */
	    PexTransformNpcToDev(pRD, cpt[i].x, cpt[i].y, &ddxpt[i].x, &ddxpt[i].y);
	}

	/* Get a normal (one of these will be set) */
	if ( qa->facetAttributes & GANormal )
	    normal = * PexQuadMeshFacetNorm(qa, pQF->i, pQF->j);
	else if ( qa->vertexAttributes & GANormal )
	    normal = * PexQuadMeshVertexNorm(qa, pQF->i, pQF->j);
	
	shade = PexCalcShade(pRD, &normal);
	miPexSetGCValues(pGC, pDraw,
			 pQF->arr->color.color.format.indexed.index,
			 shade, index);
	
	/* Draw it */
	(*GetGCValue(pGC, FillPolygon))(pDraw, pGC, Complex, CoordModeOrigin,
					vis, ddxpt);
    }

}


/*****************************************************************
 * TAG( miPexQuadMesh2DSmoothHide )
 * 
 * 2D Drawing support function for quadmesh rendering.
 * Inputs:
 *	[None]
 * Outputs:
 *	[None]
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */
miPexQuadMesh2DSmoothHide(pRD, pQF)
    pexRendererPtr pRD;
    qm_facet *pQF;
{
    pexQuadMeshPtr qa = pQF->arr->mesh;
    pexCoord4D pts[4];		/* Input points */
    pexCoord4D cpt[12];		/* clipped points */
    DDXPointRec ddxpt[12];	/* GC coordinate endpoints */
    DrawablePtr pDraw;
    GCPtr pGC;
    Pixel index;
    pexVector3D normal;
    int shade, vis, i;
    int cornerShades[4];


    if ( dbg_depth >= 0 )
	printf( "%.*sfacet_draw( 0x%x[%d,%d] )\n", dbg_depth,
		"                                                            ",
		pQF, pQF->i, pQF->j );

    pDraw = pRD -> pDraw;
    pGC =  pRD -> pGC;
    
    pts[0] = *(pexCoord4D *)PexQuadMeshVertex(qa, pQF->i, pQF->j);
    pts[1] = *(pexCoord4D *)PexQuadMeshVertex(qa, pQF->i+1, pQF->j);
    pts[2] = *(pexCoord4D *)PexQuadMeshVertex(qa, pQF->i+1, pQF->j+1);
    pts[3] = *(pexCoord4D *)PexQuadMeshVertex(qa, pQF->i, pQF->j+1);

    cornerShades[0] = PexCalcShade(
	pRD, PexQuadMeshVertexNorm(qa, pQF->i, pQF->j ) );
    cornerShades[1] = PexCalcShade(
	pRD, PexQuadMeshVertexNorm(qa, pQF->i+1, pQF->j ) );
    cornerShades[2] = PexCalcShade(
	pRD, PexQuadMeshVertexNorm(qa, pQF->i+1, pQF->j+1 ) );
    cornerShades[3] = PexCalcShade(
	pRD, PexQuadMeshVertexNorm(qa, pQF->i, pQF->j+1 ) );

    if ( !miPexPolyptSmooth( pts, cornerShades, 4, pRD,
			     pQF->arr->color.color.format.indexed.index ) )
    {
	/* Clip the polygon */
	vis = PexClipPolygon(4, pts, 12, cpt);
	if ( vis )
	{
	    for ( i = 0; i < vis; i++ )
	    {
		/* Perspective divide (finally!) */
		if ( cpt[i].w != 1.0 )
		{
		    cpt[i].x /= cpt[i].w;
		    cpt[i].y /= cpt[i].w;
		}
		/* Transform to DC */
		PexTransformNpcToDev(pRD, cpt[i].x, cpt[i].y, &ddxpt[i].x, &ddxpt[i].y);
	    }

	    miPexSetGCValues(pGC, pDraw,
			     pQF->arr->color.color.format.indexed.index,
			     cornerShades[0], index)

	    /* Draw it */
	    (*GetGCValue(pGC, FillPolygon))(pDraw, pGC, Complex, CoordModeOrigin,
					    vis, ddxpt);
	}
    }
    return( Success );
}

