/* 
 * pexmath.c - miscellaneous procs including vector, color  and cursor stuff
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#include <math.h>
#include <stdio.h>
#include "PEX.h"
#include "PEXproto.h"
#include "fillarea.h"
#include "windowstr.h"
#include "scrnintstr.h"
#include "renderer.h"
#include "colortable.h"
#include "pexmath.h"

#define  ErrorCheck    0.00000001

/*****************************************************************
 * TAG( PexUnitVector )
 * 
 *   normalize a 3D vector
 * Inputs:
 * 	*pexVector3D
 * 	*pexVector3D
 * Outputs:
 * 	returns normalized vector in second parameter
 *      returns vector (0,0,0) if the vector has no length
 * Assumptions:
 *	[None]
 * Algorithm:
 *	[None]
 */

void
PexUnitVector( invector, outvector )
    pexVector3D  *invector, *outvector;
{
    FLOAT   length;

    length = sqrt(  (invector->x * invector->x)
		  + (invector->y * invector->y)
		  + (invector->z * invector->z) );

    if ( (length < ErrorCheck) &&
         (length > -ErrorCheck) )
    {
	outvector->x = outvector->y = outvector->z = 0.0;
    }
    else
    {
	outvector->x = invector->x / length;
	outvector->y = invector->y / length;
	outvector->z = invector->z / length;
    }
}

/*****************************************************************
 * TAG( PexAvrgVertexNormals )
 * 
 * average vertex normal vectors
 * Inputs:
 * 	pFillArea3DwithDataPtr
 * 	*pexVector3D 
 * Outputs:
 * 	 average vector returned in *pexVector3D
 * Assumptions:
 *      that the fill area contains at least one vertex normal
 * Algorithm:
 *      NOTE: the returned vector is not normalized because it
 *      must be transformed to world coords first
 */

void
PexAvrgVertexNormals( pFillArea, pnormal )
    pexFillAreaWithDataPtr    pFillArea;
    pexVector3D               *pnormal;
{
    register      INT16    i;
    pexVector3D   normalizedVector;
    INT16         n;

    pnormal->x = pnormal->y = pnormal->z = 0.0;

    n = pFillArea->numVertices;
    for ( i=0; i<n; i++ )
    {
	/*  this should transform normal to world coordinates first
	 *  but since we're not using this anymore, I won't fix it
	 *  the first normal found is used instead of the average
	 */

	PexUnitVector((PexFillAreaVertexNorm( pFillArea, i )),
			&normalizedVector );
	pnormal->x += normalizedVector.x;
	pnormal->y += normalizedVector.y;
	pnormal->z += normalizedVector.z;
    }

    pnormal->x /= (FLOAT)n;
    pnormal->y /= (FLOAT)n;
    pnormal->z /= (FLOAT)n;
}

/*****************************************************************
 * TAG( PexAvrgVertexPoints )
 * 
 * average vertex points to calculate a normal vector
 * Inputs:
 * 	pFillArea3DwithDataPtr
 * 	*pexVector3D
 * Outputs:
 * 	 normal vector is returned in *pexVector3D
 * Assumptions:
 *	[None]
 * Algorithm:
 *	ptA = the first point
 *      ptB = the next point not coincident with ptA
 *      ptC = the next point not colinear with ptA & ptB determined by checking
 *      if the cross product of vector AB & AC is zero
 *      the normal vector is that cross product
 *
 *      NOTE: the returned vector is not normalized and must be
 *      transformed into world coordinates before normalizing
 */

void
PexAvrgVertexPoints( pFillArea, pnormal )
    pexFillAreaWithDataPtr   pFillArea;
    pexVector3D                *pnormal;
{
    pexCoord3D   *ptA;    /* first point in vertex list */
    pexCoord3D   *ptB;    /* second point in list non-coincident with A */
    pexCoord3D   *ptC;    /* third point in list non-colinear with A or B */
    INT16        i,n;
    pexVector3D  v1, v2;

    ptA = (PexFillAreaVertex( pFillArea, 0 ));
    ptB = (PexFillAreaVertex( pFillArea, 1 ));

    i = 2;
    n = pFillArea->numVertices;

    while ( (ptA->x == ptB->x) && (ptA->y == ptB->y) && (ptA->z == ptB->z)
	   && ( i<n ) )
    {
	ptB = (PexFillAreaVertex( pFillArea, i ));
	i++;
    }

 Point3:
    if ( i == n )
    {
	printf("avrgVertexPoints:  bad data \n");
	/*  now do what??? */
	/*  how about returning a zero vector (bum someone elses scene) */
	pnormal->x = pnormal->y = pnormal->z = 0.0;
	return;
    }

    ptC = (PexFillAreaVertex( pFillArea, i ));

    v1.x = ptB->x - ptA->x;
    v1.y = ptB->y - ptA->y;
    v1.z = ptB->z - ptA->z;

    v2.x = ptC->x - ptA->x;
    v2.y = ptC->y - ptA->y;
    v2.z = ptC->z - ptA->z;

    PexVectorCrossProduct( v1, v2, *pnormal );

    /*  if the cross product is zero, then point C was colinear with A and B
     */

#define  ABS( f )   ( ( (f) < 0 ) ?  -(f) : (f) )  

    if ( (( ABS( pnormal->x) + ABS( pnormal->y ) + ABS( pnormal->z ) ) < ErrorCheck) )
    {
	i++;
	goto Point3;
    }

}

/*****************************************************************
 * TAG( mipexCalcshade )
 * 
 *   calculate the shade from a normal vector
 * Inputs:
 * 	pexRendererPtr
 * 	pexVector3D
 * Outputs:
 * 	shade value is returned
 * Assumptions:
 *      The normals are in the world coordinate system and are normalized.
 *      The reflection attribute and light direction are normalized.
 * Algorithm:
 *	[None]
 */

PexCalcShade( pRend, pnormal )
    pexRendererPtr  pRend;
    pexVector3D     *pnormal;
{
    pexPipelineContextPtr  pPC;
    FLOAT   cos1, cos2;
    INT16   numLights;
    pexBitmask  lightState, lightMask;
    FLOAT   ambient, diffuse, specular/*, specConc*/;
    FLOAT   fshade;
    INT16   ishade;
    pexVector3D  halfwayLight, unitNormal;
    INT16   i; 

    pPC = pRend->pPC;
    PexUnitVector( pnormal, &unitNormal );
    lightState = pPC->lightState;

    /*  reflection attributes are normalized when set
     */

    ambient = pPC->reflectionAttr.ambient;
    diffuse = pPC->reflectionAttr.diffuse;
    specular = pPC->reflectionAttr.specular;
/*    specConc = pPC->reflectionAttr.specularConc;*/

    /*  light vector is normalized when initialized */

    /*  loop for each light source */

    for ( i=0, fshade=0.0, numLights=0, lightMask=(pexBitmask)1;
	     i<NUM_LIGHT_SOURCES;
		  i++, (lightMask<<=1) )
    {
	/*  add the light if it is turned on */

	if ( lightMask & lightState )
	{
	    cos1 = PexVectorDotProduct( unitNormal, pRend->lightTable[i].direction );

	    /*	 assume that the eye is at (0,0,1)
	     *	 and that the direction vector for the light is from the eye
	     *	 then the following calculation finds the vector halfway between
	     *	 the light and the eye direction
	     */

	    halfwayLight = pRend->lightTable[i].direction;
	    halfwayLight.x /= 2.0;
	    halfwayLight.y /= 2.0;
	    halfwayLight.z = (halfwayLight.z + 1.0)/2.0;

	    PexUnitVector( &halfwayLight, &halfwayLight );

	    cos2 = PexVectorDotProduct( unitNormal, halfwayLight );

	    /*	 special case  ( can't calculate pow( 0, y ) )	 */
	    if ( cos2 == 0 )
		fshade += ambient + diffuse*cos1;
	    else
		fshade += ambient + diffuse*cos1 + specular/*pow( cos2, specConc )*/;

	    numLights++;
	}
    }

    ishade = ((numLights==0) ? 0 :
	      (INT16)(fshade/(FLOAT)numLights * NUM_SHADES));

    if ( (ishade < 0) || (ishade >= NUM_SHADES) )
    {
	fprintf(stderr,"Shading error in PexCalcShade.\n");
	ishade = TRUE_COLOR_SHADE;
    }
    return( ishade ); 
}

/*****************************************************************
 * TAG( PexCalcVertexColors )
 * 
 *  calculate a color for each vertex in a fill area
 * Inputs:
 *      pexRendererPtr
 * 	pFillArea3DwithDataPtr
 *      CARD16  baseColor
 * 	INT16  colors[]
 * Outputs:
 * 	array of color indexes in second parameter
 * Assumptions:
 *	vertex normals exist in fillarea
 * Algorithm:
 *	[None]
 */

void
PexCalcVertexColors( pRend, pFillArea, baseColor, colors )
    pexRendererPtr    pRend;
    pexFillAreaWithDataPtr   pFillArea;
    CARD16   baseColor;
    INT32    colors[];
{
    INT16    i,n;
    pexVector3D normal;
    
    n = pFillArea->numVertices;

    for ( i=0; i<n; i++ )
    {
	PexTransformNorm( pRend, PexFillAreaVertexNorm( pFillArea, i ),
			 &normal );

	colors[i] = PexColorTableIndex( baseColor,
				       PexCalcShade( pRend, &normal ) );
    }
}

/*****************************************************************
 * TAG( PexCullFacet )
 * 
 * determine, from a list of a facets vertices, whether that facet should
 * be culled or not.
 *
 * Inputs:
 * 	*pexVector4D
 * Outputs:
 * 	TRUE is returned if the facet should be culled, FALSE otherwise.
 * Assumptions:
 *	Vertices have been transformed through all our favorite matrices, and
 *      now exist in npc space.
 * Algorithm:
 *	calculate the normal, and cull if z is negative.
 */

int
PexCullFacet ( pts, numPts )
    pexCoord4D *pts;
    CARD16 numPts;
{
    pexCoord4D   *ptA;    /* first point in vertex list */
    pexCoord4D   *ptB;    /* second point in list non-coincident with A */
    pexCoord4D   *ptC;    /* third point in list non-colinear with A or B */
    INT16        i;
    pexVector3D  v1, v2, v3;


    ptA = &(pts[0]);
    ptB = &(pts[1]);

    i = 2;

    while ( (ptA->x == ptB->x) && (ptA->y == ptB->y) && (ptA->z == ptB->z)
	   && ( i<numPts ) )
    {
	ptB = &(pts[i]);
	i++;
    }

 Point3:
    if ( i == numPts )
    {
	printf("PexCullFacet:  bad data \n");
	/*  We better cull this if the data is bad. Return True. */
	return (TRUE);
    }

    ptC = &(pts[i]);

    v1.x = (ptB->x/ptB->w) - (ptA->x/ptA->w);
    v2.x = (ptC->x/ptC->w) - (ptA->x/ptA->w);
    v1.y = (ptB->y/ptB->w) - (ptA->y/ptA->w);
    v2.y = (ptC->y/ptC->w) - (ptA->y/ptA->w);
    v1.z = (ptB->z/ptB->w) - (ptA->z/ptA->w);
    v2.z = (ptC->z/ptC->w) - (ptA->z/ptA->w);

    PexFacetCullingVectorCrossProduct( v1, v2, v3 );

    return ((v3.z < 0) ? TRUE : FALSE);
}
