/* 
 * Slate.c - Simple "slate" widget - provide a drawing surface.
 * 
 * Copyright 1988
 * Center for Information Technology Integration (CITI)
 * Information Technology Division
 * University of Michigan
 * Ann Arbor, Michigan
 *
 *                         All Rights Reserved
 * 
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the names of
 * CITI or THE UNIVERSITY OF MICHIGAN not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS." CITI AND THE UNIVERSITY OF
 * MICHIGAN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * NO EVENT SHALL CITI OR THE UNIVERSITY OF MICHIGAN BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

#define XtStrlen(s)		((s) ? strlen(s) : 0)

#include <X11/Xos.h>
#include <X11/IntrinsicP.h>
#include <X11/Xresource.h>
#include <X11/StringDefs.h>
#include "SlateP.h"

/* Private definitions */

static char defaultTranslations[] = 
    "<BtnDown>:   ButtonDown()\n\
     <Map>:	  MapEvent(Map)\n\
     <Unmap>:	  MapEvent(Unmap)\n\
     <Key>:	  KeyDown()";

static int DEFAULTSIZE = 100;
static int DEFAULTXY = 4;

#define Offset(field) XtOffset(SlateWidget, field)

static XtResource resources[] = {
  {XtNwidth, XtCWidth, XtRInt, sizeof(int),
	     Offset(core.width), XtRInt, (caddr_t)&DEFAULTSIZE},
  {XtNheight, XtCHeight, XtRInt, sizeof(int),
	     Offset(core.height), XtRInt, (caddr_t)&DEFAULTSIZE},
  {XtNbuttonProc, XtCCallback, XtRCallback, sizeof(caddr_t),
	     Offset(slate.button_proc), XtRCallback, NULL},
  {XtNkeyProc, XtCCallback, XtRCallback, sizeof(caddr_t),
	     Offset(slate.key_proc), XtRCallback, NULL},
  {XtNredisplayProc, XtCCallback, XtRCallback, sizeof(caddr_t),
	     Offset(slate.redisp_proc), XtRCallback, NULL},
  {XtNresizeProc, XtCCallback, XtRCallback, sizeof(caddr_t),
	     Offset(slate.resize_proc), XtRCallback, NULL},
  {XtNmapProc, XtCCallback, XtRCallback, sizeof(caddr_t),
       	     Offset(slate.map_proc), XtRCallback, NULL},
  {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
	     Offset(slate.foreground), XtRString, "black"},
  {XtNhighlight, XtCForeground, XtRPixel, sizeof(Pixel),
       	     Offset(slate.highlight), XtRString, "red"},
  {XtNdepthCue, XtCForeground, XtRPixel, sizeof(Pixel),
       	     Offset(slate.depth_cue), XtRString, "light gray"},
  {XtNslateCursor, XtCCursor, XtRCursor, sizeof(Cursor),
	     Offset(slate.cursor), XtRString, "crosshair"},
  {XtNfont,  XtCFont, XtRFontStruct, sizeof(XFontStruct *),
       	     Offset(slate.font),XtRString, "Fixed"},
  {XtNlabel, XtCLabel, XtRString, sizeof(String),
       	     Offset(slate.label), XtRString, NULL},
  {XtNlabelX, XtCX, XtRInt, sizeof(int),
       	     Offset(slate.x), XtRInt, (caddr_t)&DEFAULTXY},
  {XtNlabelY, XtCY, XtRInt, sizeof(int),
       	     Offset(slate.y), XtRInt, (caddr_t)&DEFAULTXY},
};

static void ClassInitialize();
static void Initialize();
static void Realize();
static void Resize();
static void Redisplay();
static Boolean SetValues();

static void ButtonDown(), MapEvent(), KeyDown();

static XtActionsRec actions[] = {
	{"ButtonDown",		ButtonDown},
	{"MapEvent",		MapEvent},
    	{"KeyDown",		KeyDown},
    	{NULL, NULL}
};

static SlateClassRec slateClassRec = {
/* core fields */
    /* superclass       */      (WidgetClass) &widgetClassRec,
    /* class_name       */      "Slate",
    /* size             */      sizeof(SlateRec),
    /* class_initialize	*/	ClassInitialize,
    /* class_part_init  */	NULL,
    /* class_inited	*/	FALSE,
    /* initialize       */      Initialize,
    /* initialize_hook  */	NULL,
    /* realize          */      Realize,
    /* actions          */      actions,
    /* num_actions	*/	XtNumber(actions),
    /* resources        */      resources,
    /* num_resources    */      XtNumber(resources),
    /* xrm_class        */      NULLQUARK,
    /* compress_motion	*/	TRUE,
    /* compress_exposure*/	TRUE,
    /* compress_enterleave*/	TRUE,
    /* visible_interest */      FALSE,
    /* destroy          */      NULL,
    /* resize           */      Resize,
    /* expose           */      Redisplay,
    /* set_values       */      SetValues,
    /* set_values_hook  */	NULL,
    /* set_values_almost */	XtInheritSetValuesAlmost,
    /* get_values_hook  */	NULL,
    /* accept_focus     */      NULL,
    /* version          */	XtVersion,
    /* callback_private */      NULL,
    /* tm_table         */      defaultTranslations,
    /* query_geometry	*/	NULL,
};

WidgetClass  slateWidgetClass = (WidgetClass)&slateClassRec;

static void ClassInitialize()
{
    /* No work? */
}

/* Clear the area covered by the label for this slate */
static void ClearLabel( w )
   SlateWidget w;
{
    XCharStruct extnt;
    int dir, ascent, descent;
    int x, y;

    if ( w->slate.label && XtWindow( w ) )
    {
	XTextExtents( w->slate.font, w->slate.label, w->slate.label_len,
		      &dir, &ascent, &descent, &extnt );
	x = w->slate.x - extnt.lbearing;
	if ( x < 0 ) x = 0;
	y = w->slate.y + ascent - extnt.ascent;
	if ( y < 0 ) y = 0;
	XClearArea( XtDisplay( w ), XtWindow( w ),
		    x, y,
		    extnt.rbearing + extnt.lbearing,
		    extnt.descent + extnt.ascent,
		    False );
    }
}

/* Draw the label for this slate */
static void DrawLabel( w )
   SlateWidget w;
{
    if ( w->slate.label && XtWindow( w ) )
    {
	XDrawImageString( XtDisplay( w ), XtWindow( w ), w->slate.gc,
			  w->slate.x,
			  w->slate.y + w->slate.font->max_bounds.ascent,
			  w->slate.label, w->slate.label_len );
    }
}
			

/* ARGSUSED */
static void Initialize( request, new )
   Widget request;		/* what the client asked for */
   Widget new;			/* what we're going to give him */
{
    SlateWidget w = (SlateWidget) new;
    XGCValues gcValues;

    if ( w->slate.label != NULL )
    {
	w->slate.label_len = XtStrlen(w->slate.label);
	w->slate.label = strcpy( XtMalloc(w->slate.label_len + 1),
				       w->slate.label );
    }

    gcValues.font = w->slate.font->fid;
    gcValues.foreground = w->slate.foreground;
    gcValues.background = w->core.background_pixel;

    w->slate.gc = XtGetGC( new,
			   GCForeground | GCBackground | GCFont,
			   &gcValues);

}

static void Realize( gw, valueMask, attributes )
   Widget gw;
   Mask *valueMask;
   XSetWindowAttributes *attributes;
{
    SlateWidget w = (SlateWidget) gw;

    attributes->cursor = w->slate.cursor;
    *valueMask |= CWCursor;
    
    XtCreateWindow( gw, InputOutput, (Visual *)CopyFromParent,
		    *valueMask, attributes );
}


/* ARGSUSED */
static Boolean SetValues( current, request, desired )
   Widget current,		/* what I am */
          request,		/* what he wants me to be */
          desired;		/* what I will become */
{
    SlateWidget w = (SlateWidget) current;
    SlateWidget rw = (SlateWidget) request;
    SlateWidget dw = (SlateWidget) desired;
    Boolean redraw = FALSE;

    if ( w->slate.label != dw->slate.label )
    {
	ClearLabel( w );
	XtFree( (char *)w->slate.label );
    }

    if ( dw->slate.label != NULL )
    {
	dw->slate.label_len = XtStrlen(dw->slate.label);
	dw->slate.label = strcpy( XtMalloc(dw->slate.label_len+1),
				  dw->slate.label );
	DrawLabel( dw );
    }

    if (w->slate.foreground != rw->slate.foreground ||
	w->core.background_pixel != rw->core.background_pixel)
        redraw = TRUE;

    if (w->slate.font != dw->slate.font)
	redraw = TRUE;

    return( redraw );
}

/* ARGSUSED */
static void Resize( gw )
   Widget gw;
{
    /* Call the resize callback */
    XtCallCallbacks( gw, XtNresizeProc, NULL );

    /* Then redisplay */
    Redisplay( gw, (XEvent*)NULL );
}


/* ARGSUSED */
static void Redisplay( gw, event, region )
   Widget gw;
   XEvent *event;		/* unused, NULL if called from Resize() */
   Region region;		/* NULL if called from Resize() */
{
    SlateWidget w = (SlateWidget) gw;

    /* Just call the redisplay callback */
    XtCallCallbacks( gw, XtNredisplayProc, region );
    
    /* Then draw the label */
    DrawLabel( w );
}

/* ARGSUSED */
static void
ButtonDown( gw, event, params, num_params )
Widget gw;
XEvent *event;
String *params;			/* Not used */
Cardinal *num_params;		/* should be 0 */
{
    XButtonPressedEvent *bev = (XButtonPressedEvent *)event;
    int parms[3];

    /* Figure out button and modifier (shift) */
    parms[0] = bev->button;
    if ( bev->state & ShiftMask )
	parms[0] = -parms[0];

    parms[1] = bev->x;
    parms[2] = bev->y;

    /* Call button callback */
    XtCallCallbacks( gw, XtNbuttonProc, (caddr_t)parms );
}

/* ARGSUSED */
static void
KeyDown( gw, event, params, num_params )
Widget gw;
XEvent *event;
String *params;			/* Not used */
Cardinal *num_params;		/* should be 0 */
{
    /* Call key callback */
    XtCallCallbacks( gw, XtNkeyProc, (caddr_t)event );
}

/* ARGSUSED */
static void
MapEvent( gw, event, params, num_params )
Widget gw;
XEvent *event;
String *params;			/* Map or Unmap */
Cardinal *num_params;		/* Should be 1 */
{
    int parm;

    if ( *params[0] == 'M' )
	parm = 1;		/* Mapping */
    else
	parm = 0;		/* Unmapping */

    /* Call map callbacks */
    XtCallCallbacks( gw, XtNmapProc, (caddr_t)&parm );
}
