
/*

	Copyright (c) 1986 	Chris Guthrie

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation.  No representations are made about the
suitability of this software for any purpose.  It is
provided "as is" without express or implied warranty.

*/

/*
 * X11 support and other enhancements added by Jeff Weinstein
 * (jeff@polyslo.calpoly.edu).  Please send all comments, bug
 * reports, fixes, suggestions regarding this version of XTrek
 * to me.  
 */

static char RCSID[] = "$Header: /blackbird/home/jeff/TAPE2/xtrek.new/RCS/rmove.c,v 3.1 88/09/20 00:44:44 jeff Exp $";

#include <X11/Xlib.h>
#include <stdio.h>
#include <signal.h>
#include <math.h>
#include "defs.h"
#include "struct.h"
#include "data.h"

#define SIZEOF(s)		(sizeof (s) / sizeof (*(s)))
#define AVOID_TIME		4
#define AVOID_CLICKS		200

#define NORMALIZE(d) 		((((d) % 256) + 256) % 256)

#define E_INTRUDER	0x01
#define E_TSHOT		0x02
#define E_PSHOT 	0x04

struct Enemy {
    int e_info;
    int e_dist;
    unsigned char e_course;	/* course to enemy */
    unsigned char e_tcourse;	/* torpedo intercept course to enemy */
    unsigned int e_flags;
};

#define NO_PLAYERS	-1

int timer;
static int avoidTime;
static unsigned short tok;

extern int debug;
extern int hostile;
extern int sticky;
extern int practice;

unsigned char	getcourse();

rmove()
{
    register int i;
    register int burst;
    register int numHits, tDir;
    int		avDir;
    extern struct Enemy *get_nearest();
    struct Enemy *enemy_buf;
    struct player *enemy;
    static int	clock = 0;
    static int	avoid[2] = { -32, 32 };
    int no_cloak;

    clock++;
    /* Check that I'm alive */
    if (me->p_status == PEXPLODE) {
	signal(SIGALRM, SIG_IGN);
	while (me->p_status == PEXPLODE)
	    ;
	while (me->p_ntorp > 0)
	    ;
	me->p_status = PFREE;
	exit(0);
    }
    if (me->p_status == PDEAD) {
	signal(SIGALRM, SIG_IGN);
	me->p_status = PFREE;
	exit(0);
    }

    /* keep ghostbuster away */
    me->p_ghostbuster = 0;

    /* Find an enemy */

    enemy_buf = get_nearest();

    if (enemy_buf == (struct Enemy *) NO_PLAYERS) { /* no more players. wait 1 minute. */
	if (!sticky) {
	    if (timer == 0)
		timer = me->p_updates + 600;
	    if (me->p_updates >= timer) {
		signal(SIGALRM, SIG_IGN);
		me->p_status = PFREE;
		exit(0);
	    }
	}
	if (do_repair()) {
	    return;
	}
	go_home(0);
	if (debug)
	    fprintf(stderr, "%d) No players in game.\n", me->p_no);
	return;
    }
    else if (enemy_buf == 0) {	 /* no one hostile */
	if (debug)
	    fprintf(stderr, "%d) No hostile players in game.\n", me->p_no);
	if (do_repair()) {
	    return;
	}
	go_home(0);
	timer = 0;
	return;
    }
    else {		/* Someone to kill */
	enemy = &players[enemy_buf->e_info];
	timer = 0;
	if (debug)
	    fprintf(stderr, "%d) noticed %d\n", me->p_no, enemy->p_no);
    }

/* Algorithm:
** We have an enemy.
** First priority: shoot at target in range.
** Second: Dodge torps.
** Third: Get away if we are damaged.
** Fourth: repair.
** Fifth: attack.
*/

/*
** If we are a practice robot, we will do all but the second.  One
** will be modified to shoot poorly and not use phasers.
**/
    /* Fire weapons!!! */
    /*
    ** get_nearest() has already determined if torpedoes and phasers
    ** will hit.  It has also determined the courses which torps and
    ** phasers should be fired.  If so we will go ahead and shoot here.
    ** We will lose repair and cloaking for the rest of this interrupt.
    ** if we fire here.
    */

    if (practice) {
	no_cloak = 1;
	if (enemy_buf->e_flags & E_TSHOT) {
	    if (debug)
		fprintf(stderr, "%d) firing torps\n", me->p_no);
	    for (burst = 0; (burst < 3) && (me->p_ntorp < MAXTORP); burst++) {
		tok += 11;
		ntorp(enemy_buf->e_tcourse, TMOVE, tok);
	    }
	}
    }
    else {
	no_cloak = 0;
	if (enemy_buf->e_flags & E_TSHOT) {
	    if (debug)
		fprintf(stderr, "%d) firing torps\n", me->p_no);
	    for (burst = 0; (burst < 2) && (me->p_ntorp < MAXTORP); burst++) {
		repair_off();
		cloak_off();
		tok += 11;
		ntorp(enemy_buf->e_tcourse, TSTRAIGHT, tok);
		no_cloak++;
	    }
	}
	if (enemy_buf->e_flags & E_PSHOT) {
	    if (debug)
		fprintf(stderr, "%d) phaser firing\n", me->p_no);
	    no_cloak++;
	    repair_off();
	    cloak_off();
	    phaser(enemy_buf->e_course);
	}
    }

    /* Avoid torps */
    /*
    ** This section of code allows robots to avoid torps.
    ** Within a specific range they will check to see if
    ** any of the 'closest' enemies torps will hit them.
    ** If so, they will evade for four updates.
    ** Evading is all they will do for this round, other than shooting.
    */

    if (!practice) {
	if (enemy->p_ntorp < 5) {
	    if ((enemy_buf->e_dist < 15000) || (avoidTime > 0)) {
		numHits = projectDamage(enemy->p_no, &avDir);
		if (debug) {
		    fprintf(stderr, "%d hits expected from %d from dir = %d\n",
			    numHits, enemy->p_no, avDir);
		}
		if (numHits == 0) {
		    if (--avoidTime > 0) {	/* we may still be avoiding */
			if (angdist(me->p_desdir, me->p_dir) > 64)
			    me->p_desspeed = 3;
			else
			    me->p_desspeed = 5;
			return;
		    }
		} else {
		    /*
		     * Actually avoid Torps
		     */ 
		    avoidTime = AVOID_TIME;
		    tDir = avDir - me->p_dir;
		    /* put into 0->255 range */
		    tDir = NORMALIZE(tDir);
		    if (debug)
			fprintf(stderr, "mydir = %d avDir = %d tDir = %d q = %d\n",
			    me->p_dir, avDir, tDir, tDir / 64);
		    switch (tDir / 64) {
		    case 0:
		    case 1:
			    me->p_desdir = NORMALIZE(avDir + 64);
			    break;
		    case 2:
		    case 3:
			    me->p_desdir = NORMALIZE(avDir - 64);
			    break;
		    }
		    if (!no_cloak)
			cloak_on();

		    if (angdist(me->p_desdir, me->p_dir) > 64)
			me->p_desspeed = 3;
		    else
			me->p_desspeed = 5;
			
		    shield_up();
		    if (debug)
			fprintf(stderr, "evading to dir = %d\n", me->p_desdir);
		    return;
		}
	    }
	}

	/*
	** Trying another scheme.
	** Robot will keep track of the number of torps a player has
	** launched.  If they are greater than say four, the robot will
	** veer off immediately.  Seems more humanlike to me.
	*/

	else if (enemy_buf->e_dist < 15000) {
	    if (--avoidTime > 0) {	/* we may still be avoiding */
		if (angdist(me->p_desdir, me->p_dir) > 64)
		    me->p_desspeed = 3;
		else
		    me->p_desspeed = 5;
		return;
	    }
	    if (random() % 2) {
		me->p_desdir = NORMALIZE(enemy_buf->e_course - 64);
		avoidTime = AVOID_TIME;
	    }
	    else {
		me->p_desdir = NORMALIZE(enemy_buf->e_course + 64);
		avoidTime = AVOID_TIME;
	    }
	    if (angdist(me->p_desdir, me->p_dir) > 64)
		me->p_desspeed = 3;
	    else
		me->p_desspeed = 5;
	    shield_up();
	    return;
	}
    }
	    
    /* Run away */
    /*
    ** The robot has taken damage.  He will now attempt to run away from
    ** the closest player.  This obviously won't do him any good if there
    ** is another player in the direction he wants to go.
    ** Note that the robot will not run away if he dodged torps, above.
    ** The robot will lower his shields in hopes of repairing some damage.
    */

    if (me->p_damage > 0 && enemy_buf->e_dist < 13000) {
	if (me->p_etemp > 900)		/* 90% of 1000 */
	    me->p_desspeed = 5;
	else
	    me->p_desspeed = 6;
	if (!no_cloak)
	    cloak_on();
	repair_off();
	shield_down();
	me->p_desdir = enemy_buf->e_course - 128;
	if (debug)
	    fprintf(stderr, "%d(%d)(%d/%d) running from %c%d %16s damage (%d/%d) dist %d\n",
		me->p_no,
		(int) me->p_kills,
		me->p_damage,
		me->p_shield,
		teamlet[enemy->p_team],
		enemy->p_no,
		enemy->p_login,
		enemy->p_damage,
		enemy->p_shield,
		enemy_buf->e_dist);
	return;
    }

    /* Repair if necessary (we are safe) */
    /*
    ** The robot is safely away from players.  It can now repair in peace.
    ** It will try to do so now.
    */

    if (do_repair()) {
	return;
    }

    /* Attack. */
    /*
    ** The robot has nothing to do.  It will check and see if the nearest
    ** enemy fits any of its criterion for attack.  If it does, the robot
    ** will speed in and deliver a punishing blow.  (Well, maybe)
    */

    if ((enemy_buf->e_flags & E_INTRUDER) || (enemy_buf->e_dist < 15000)
	|| (hostile)) {
	if ((!no_cloak) && (enemy_buf->e_dist < 10000))
	    cloak_on();
	shield_up();
	if (debug)
	    fprintf(stderr, "%d(%d)(%d/%d) attacking %c%d %16s damage (%d/%d) dist %d\n",
		me->p_no,
		(int) me->p_kills,
		me->p_damage,
		me->p_shield,
		teamlet[enemy->p_team],
		enemy->p_no,
		enemy->p_login,
		enemy->p_damage,
		enemy->p_shield,
		enemy_buf->e_dist);

	if (enemy_buf->e_dist < 15000) {
	    me->p_desdir = enemy_buf->e_course + 
		    avoid[(clock / AVOID_CLICKS) % SIZEOF(avoid)];
	    if (angdist(me->p_desdir, me->p_dir) > 64)
		me->p_desspeed = 3;
	    else
		me->p_desspeed = 4;
	}
	else {
	    me->p_desdir = enemy_buf->e_course;
	    if (angdist(me->p_desdir, me->p_dir) > 64)
		me->p_desspeed = 3;
	    else if (me->p_etemp > 900)		/* 90% of 1000 */
		me->p_desspeed = 5;
	    else
		me->p_desspeed = 6;
	}
    }
    else {
	go_home(enemy_buf);
    }
}

unsigned char
getcourse(x, y)
int x, y;
{
#ifdef SYSATAN2
	return((unsigned char) (atan2((double) (x - me->p_x),
	    (double) (me->p_y - y)) / 3.14159 * 128.));
#else
	return( myatan2( x - me->p_x, me->p_y - y ) );
#endif
}

struct {
    int x;
    int y;
} center[] = { {0, 0},
		{GWIDTH / 4, GWIDTH * 3 / 4},		/* Fed */
		{GWIDTH / 4, GWIDTH / 4},		/* Rom */
		{0, 0},
		{GWIDTH * 3 / 4, GWIDTH  / 4},		/* Kli */
		{0, 0},
		{0, 0},
		{0, 0},
		{GWIDTH * 3 / 4, GWIDTH * 3 / 4}};	/* Ori */

/* This function means that the robot has nothing better to do.
   If there are hostile players in the game, it will try to get
   as close to them as it can, while staying in it's on space.
   Otherwise, it will head to the center of its own space.
*/
go_home(ebuf)
struct Enemy *ebuf;
{
    int x, y, speed;
    double dx, dy;
    int tdist;
    struct player *j;

    if (ebuf == 0) {  /* No enemies */
	if (debug)
	    fprintf(stderr, "%d) No enemies\n", me->p_no);
	x = center[me->p_team].x;
	y = center[me->p_team].y;
    }
    else {	/* Let's get near him */
	j = &players[ebuf->e_info];
	x = j->p_x;
	y = j->p_y;
	switch (me->p_team) {
	    case FED:
		if (x > (GWIDTH/2) - 5000)
		    x = (GWIDTH/2) - 5000;
		if (y < (GWIDTH/2) + 5000)
		    y = (GWIDTH/2) + 5000;
		break;
	    case ROM:
		if (x > (GWIDTH/2) - 5000)
		    x = (GWIDTH/2) - 5000;
		if (y > (GWIDTH/2) - 5000)
		    y = (GWIDTH/2) - 5000;
		break;
	    case KLI:
		if (x < (GWIDTH/2) + 5000)
		    x = (GWIDTH/2) + 5000;
		if (y > (GWIDTH/2) - 5000)
		    y = (GWIDTH/2) - 5000;
		break;
	    case ORI:
		if (x < (GWIDTH/2) + 5000)
		    x = (GWIDTH/2) + 5000;
		if (y < (GWIDTH/2) + 5000)
		    y = (GWIDTH/2) + 5000;
		break;
	}
    }
    if (debug)
	fprintf(stderr, "%d) moving towards (%d/%d)\n",
	    me->p_no, x, y);

    /* Note that I've decided that robots should never stop moving.
    ** It makes them too easy to kill
    */

    me->p_desdir = getcourse(x, y);
    if (angdist(me->p_desdir, me->p_dir) > 64)
	me->p_desspeed = 3;
    else if (me->p_etemp > 900)		/* 90% of 1000 */
	me->p_desspeed = 5;
    else {
	dx = x - me->p_x;
	dy = y - me->p_y;
	me->p_desspeed = (hypot(dx, dy) / 5000) + 3;
    }
    cloak_off();
}

projectDamage(eNum, dirP)
	int	*dirP;
{
	register int		i, j, numHits = 0, mx, my, tx, ty, dx, dy;
	double			tdx, tdy, mdx, mdy;
	register struct torp	*t;

	*dirP = 0;
	for (i = 0, t = &torps[eNum * MAXTORP]; i < MAXTORP; i++, t++) {
		if (t->t_status == TFREE)
			continue;
		tx = t->t_x; ty = t->t_y;
		mx = me->p_x; my = me->p_y;
		tdx = (double) t->t_speed * Cos[t->t_dir] * WARP1;
		tdy = (double) t->t_speed * Sin[t->t_dir] * WARP1;
		mdx = (double) me->p_speed * Cos[me->p_dir] * WARP1;
		mdy = (double) me->p_speed * Sin[me->p_dir] * WARP1;
		for (j = t->t_fuse; j > 0; j--) {
			tx += tdx; ty += tdy;
			mx += mdx; my += mdy;
			dx = tx - mx; dy = ty - my;
			if (ABS(dx) < EXPDIST && ABS(dy) < EXPDIST) {
				numHits++;
				*dirP += t->t_dir;
				break;
			}
		}
	}
	if (numHits > 0)
		*dirP /= numHits;
	return (numHits);
}

struct Enemy ebuf;

struct Enemy *
get_nearest()
{
    int pcount = 0;
    register int i;
    register struct player *j;
    int intruder = 0;
    int tdist;
    double dx, dy;

    /* Find an enemy */
    ebuf.e_info = me->p_no;
    ebuf.e_dist = GWIDTH + 1;

    pcount = 0;  /* number of human players in game */

    /* avoid dead slots, me, other robots (which aren't hostile) */
    for (i = 0, j = &players[i]; i < MAXPLAYER; i++, j++) {
	if ((j->p_status != PALIVE) || (j == me) ||
	    ((j->p_flags & PFROBOT) && (!hostile)))
	    continue;
	else
	    pcount++;	/* Other players in the game */
	if (((j->p_swar | j->p_hostile) & me->p_team)
		|| ((me->p_swar | me->p_hostile) & j->p_team)) {
	    /* We have an enemy */
	    /* Get his range */
	    dx = j->p_x - me->p_x;
	    dy = j->p_y - me->p_y;
	    tdist = hypot(dx, dy);

	    /* Check to see if ship is in our space. */
	    switch (me->p_team) {
		case FED:
		    intruder = ((j->p_x < GWIDTH/2) && (j->p_y > GWIDTH/2));
		    break;
		case ROM:
		    intruder = ((j->p_x < GWIDTH/2) && (j->p_y < GWIDTH/2));
		    break;
		case KLI:
		    intruder = ((j->p_x > GWIDTH/2) && (j->p_y < GWIDTH/2));
		    break;
		case ORI:
		    intruder = ((j->p_x > GWIDTH/2) && (j->p_y > GWIDTH/2));
		    break;
	    }

	    if (tdist < ebuf.e_dist) {
		ebuf.e_info = i;
		ebuf.e_dist = tdist;
		if (intruder)
		    ebuf.e_flags |= E_INTRUDER;
		else
		    ebuf.e_flags &= ~(E_INTRUDER);
	    }
	}
    }
    if (pcount == 0)
	return ((struct Enemy *) NO_PLAYERS);	/* no players in game */
    else if (ebuf.e_info == me->p_no)
	return (0);			/* no hostile players in the game */
    else {
	j = &players[ebuf.e_info];

	/* Get torpedo course to nearest enemy */
	ebuf.e_flags &= ~(E_TSHOT);
	for (i = 0; i < 50; i++) {
	    double  he_x, he_y, area;

	    he_x = j->p_x + Cos[j->p_dir] * j->p_speed * i * WARP1;
	    he_y = j->p_y + Sin[j->p_dir] * j->p_speed * i * WARP1;
	    area = i * me->p_ship.s_torpspeed * WARP1;
	    if (hypot(he_x - me->p_x, he_y - me->p_y) < area) {
		ebuf.e_flags |= E_TSHOT;
		ebuf.e_tcourse = getcourse((int) he_x, (int) he_y);
		break;
	    }
	}
	/*
	if (debug)
	    fprintf(stderr, "torpedo course to enemy %d is %d (%d) - %s\n", 
		    j->p_no,
		    (int) ebuf.e_tcourse,
		    (int) ebuf.e_tcourse * 360 / 256,
		    (ebuf.e_flags & E_TSHOT) ? "aiming to hit" :
		    "no hit possible");
	*/

	/* Get phaser shot status */
	if (ebuf.e_dist < 5000)
		ebuf.e_flags |= E_PSHOT;
	else
		ebuf.e_flags &= ~(E_PSHOT);

	/* get course info */
	ebuf.e_course = getcourse(j->p_x, j->p_y);
	/*
	if (debug)
	    fprintf(stderr, "Set course to enemy is %d (%d)\n",
		    (int)ebuf.e_course, 
		    (int) ebuf.e_course * 360 / 256);
	*/

	return (&ebuf);
    }
}

do_repair()
{
/* Repair if necessary (we are safe) */

    if (me->p_damage > 0) {
	me->p_desspeed = 0;
	cloak_off();
	shield_down();
	me->p_desspeed = 0;
	if (me->p_speed == 0)
	    repair();
	if (debug)
	    fprintf(stderr, "%d) repairing damage at %d\n",
		me->p_no,
		me->p_damage);
	return(1);
    }
    else {
	return (0);
    }
}
