/*
 * xchcursor - change cursor of selected window by specified interval
 *
 * Copyright 1988 Software Research Associates, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Software Research Associates not be 
 * used in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  Software Research Associates 
 * makes no representations about the suitability of this software for any 
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 * Author:  Makoto Ishisone
 *          original was written by toshikochyan,
 *          ported to X11 by kato@cs.titech.junet
 *
 *
 *	xchcursor.c	ver 1.5
 *		change cursor of selected window by specified interval
 *
 *	Usage:
 *	xchcursor [-rv] [-fg color] [-bg color] [-window ID] [-select ]
 *			[-interval N] [host:display] cursor mask ...
 *
 *		-rv	reverse mode
 *		-fg	foreground color
 *		-bg	background color
 *		-interval
 *			changing interval (msec)
 *
 *		-window ID
 *			window ID
 *		-select
 *			select by mouse
 *
 *		-display display name (X11)
 */

#ifdef X11
#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/Xutil.h>
#else
#include <X/Xlib.h>
#endif
#include <stdio.h>
#include <sys/time.h>
#include <signal.h>
#include <strings.h>

#ifdef X11
#include <X11/bitmaps/target>
#else
#include "target.cursor"
#include "target_mask.cursor"
#endif

#ifdef X11
Display	*dpy;
XColor	fg_color, bg_color, junk_color;
#endif

Cursor	*cursor;
int	ncursor;
Window	target = 0;

char	*cmd;

main(ac, av)
int	ac;
char	**av;
{
#ifdef X11
	Pixmap	*c_bits, *m_bits;
#else
	short	*c_bits, *m_bits;
#endif
	char	*display = NULL;
	char	*fgc = NULL;
	char	*bgc = NULL;
#ifdef X11
	unsigned long	c_col;
	unsigned long	m_col;
#else
	int	c_col = BlackPixel;
	int	m_col = WhitePixel;
	Color	color;
#endif
	struct itimerval	itime;
	int	interval = 1000;
	int	width, height, x_hot, y_hot;
	int	status;
	int	i;
	int	sel = 0;
	int	reverse = 0;
	int	changeCursor();

	cmd = *av;
	ac--, av++;
	while (ac > 0 && **av == '-') {
		if (!strcmp(*av, "-rv")) {
			reverse = 1;
		} else if (!strcmp(*av, "-interval")) {
			interval = atoi(*++av);
			ac--;
		} else if (!strcmp(*av, "-window")) {
#ifdef	sun
			target = strtol(*++av, 0, 0);
#else
			target = atoi(*++av);
#endif
			ac--;
		} else if (!strcmp(*av, "-select")) {
			sel++;
		} else if (!strcmp(*av, "-fg")) {
			fgc = *++av;
			ac--;
		} else if (!strcmp(*av, "-bg")) {
			bgc = *++av;
			ac--;
#ifdef X11
		} else if (!strcmp(*av, "-display")) {
			display = *++av;
			ac--;
#endif
		} else {
			usage();
		}
		ac--, av++;
	}

	if (ac <= 0)
		usage();

#ifndef X11
	for (i = 0; i < ac; i++) {
		if (index(av[i], ':')) {
			display = av[i];
			i++;
			while (i < ac) {
				av[i - 1] = av[i];
				i++;
			}
			ac--;
			break;
		}
	}
#endif

	if (ac < 1)
		usage();
	if (ac & 1) {
		fprintf(stderr, "wrong file number (odd)\n");
		exit(1);
	}

	ncursor = ac / 2;
	cursor = (Cursor *)malloc(sizeof(Cursor) * ncursor);
	if (cursor == NULL) {
		fprintf(stderr, "too many cursors\n");
		exit(1);
	}

#ifdef X11
	if ((dpy = XOpenDisplay(display)) == NULL) {
#else
	if (XOpenDisplay(display) == NULL) {
#endif
		fprintf(stderr, "can't open display '%s'\n",
			XDisplayName(display));
		exit(1);
	}

#ifdef X11
	if (sel) {
		Cursor	tcurs;
		XEvent	event;
		Window  root_win, parent_win, *children;
		int     nchild ;
		char	*name;
	
		if (!(tcurs = XCreateFontCursor(dpy, XC_target))) {
			fprintf(stderr, "can't create cursor\n");
			exit(1);
		}

		if (XGrabPointer(dpy, DefaultRootWindow(dpy), False,
			ButtonPressMask|ButtonReleaseMask, GrabModeSync,
			GrabModeAsync, None, tcurs, CurrentTime)
			!= GrabSuccess) {
			fprintf(stderr, "can't grab mouse\n");
			exit(1);
		}
		XAllowEvents(dpy, SyncPointer, CurrentTime);
		XWindowEvent(dpy, DefaultRootWindow(dpy),
			ButtonPressMask|ButtonReleaseMask, &event);
/*		XNextEvent(dpy, &event);	*/
		target = event.xbutton.subwindow;
		XUngrabPointer(dpy, CurrentTime);
		XFlush(dpy);
		if (target != 0 &&
			XQueryTree(dpy, target, &root_win,
					&parent_win, &children, &nchild) &&
			nchild > 0) {
			/* select one of the children if any */
			while (nchild--) {
				if (XFetchName(dpy, children[nchild], &name))
					target = children[nchild] ;
			}
		}
	}
	if (target == (Window)0)
		target = DefaultRootWindow(dpy);
#else
	if (sel) {
		Cursor	tcurs;
		XEvent	event;

		if (!(tcurs = XCreateCursor(target_width, target_height,
				target_bits, target_mask_bits, 8, 8,
				BlackPixel, WhitePixel, GXcopy))) {
			fprintf(stderr, "can't create cursor\n");
			exit(1);
		}

		if (!XGrabMouse(RootWindow, tcurs, ButtonPressed)) {
			fprintf(stderr, "can't grab mouse\n");
			exit(1);
		}
		XNextEvent(&event);
		target = event.subwindow;
		XUngrabMouse();
	}
	if (target == 0)
		target = RootWindow;
#endif

#ifdef X11
	if (reverse) {
		fg_color.pixel = WhitePixel(dpy, DefaultScreen(dpy));
		bg_color.pixel = BlackPixel(dpy, DefaultScreen(dpy));
	} else {
		fg_color.pixel = BlackPixel(dpy, DefaultScreen(dpy));
		bg_color.pixel = WhitePixel(dpy, DefaultScreen(dpy));
	}
	XQueryColor(dpy, DefaultColormap(dpy, DefaultScreen(dpy)),
		    &fg_color);
	XQueryColor(dpy, DefaultColormap(dpy, DefaultScreen(dpy)),
		&bg_color);
	if (fgc != NULL) {
		XAllocNamedColor(dpy, DefaultColormap(dpy, DefaultScreen(dpy)),
				 fgc, &fg_color, &junk_color);
	}
	if (bgc != NULL) {
		XAllocNamedColor(dpy, DefaultColormap(dpy, DefaultScreen(dpy)),
				 bgc, &bg_color, &junk_color);
	}
#else
	if (reverse) {
		c_col = WhitePixel;
		m_col = BlackPixel;
	}
	if (fgc != NULL &&
	    XParseColor(fgc, &color) > 0 && XGetHardwareColor(&color) > 0)
		c_col = color.pixel;
	if (bgc != NULL &&
	    XParseColor(bgc, &color) > 0 && XGetHardwareColor(&color) > 0)
		m_col = color.pixel;
#endif

	for (i = 0; i < ncursor; i++) {
#ifdef X11
		status = XReadBitmapFile(dpy, DefaultRootWindow(dpy),
			*av++, &width, &height, &c_bits, &x_hot, &y_hot);
		if (status != BitmapSuccess) {
			fprintf(stderr, "can't read bitmap %s\n", *(av-1));
			exit(1);
		}
		status = XReadBitmapFile(dpy, DefaultRootWindow(dpy),
			*av++, &width, &height, &m_bits,
			(int *)NULL, (int *)NULL);
		if (status != BitmapSuccess) {
			fprintf(stderr, "can't read bitmap %s\n", *(av-1));
			exit(1);
		}
		cursor[i] = XCreatePixmapCursor(dpy, c_bits, m_bits,
			&fg_color, &bg_color,
			(unsigned int)x_hot, (unsigned int)y_hot);
		if (cursor[i] == 0) {
			fprintf(stderr, "can't create cursor #%d\n", i + 1);
			exit(1);
		}
		XFreePixmap(dpy, c_bits);
		XFreePixmap(dpy, m_bits);
#else
		status = XReadBitmapFile(*av++, &width, &height, &c_bits,
					 &x_hot, &y_hot);
		if (status <= 0) {
			fprintf(stderr, "can't read bitmap %s\n", *(av - 1));
			exit(1);
		}
		status = XReadBitmapFile(*av++, &width, &height, &m_bits,
					 (int *)NULL, (int *)NULL);
		if (status <= 0) {
			fprintf(stderr, "can't read bitmap\n");
			exit(1);
		}
		cursor[i] = XCreateCursor(width, height,
			c_bits, m_bits, x_hot, y_hot,
			c_col, m_col, GXcopy);
		if (cursor[i] == 0) {
			fprintf(stderr, "can't create cursor #%d\n", i + 1);
			exit(1);
		}
#endif
	}

	changeCursor();

	if (ncursor == 1)
		exit(0);

	(void)signal(SIGALRM, changeCursor);

	itime.it_value.tv_sec = interval / 1000;
	itime.it_value.tv_usec = (interval % 1000) * 1000;
	itime.it_interval = itime.it_value;
	if (setitimer(ITIMER_REAL, &itime, (struct itimerval *)NULL) < 0) {
		perror("setitimer");
		exit(1);
	}

	for (;;) {
		sigpause(0);
	}

	/* NOTREACHED */
}

changeCursor()
{
	static int	count;

#ifdef X11
	XDefineCursor(dpy, target, cursor[count++ % ncursor]);
	XFlush(dpy);
#else
	XDefineCursor(target, cursor[count++ % ncursor]);
	XFlush();
#endif
}

usage()
{
	fprintf(stderr,
#ifdef X11
	"Usage: %s [-rv] [-fg color] [-bg color] [-window ID] [-select] [-interval N(msec)] [-display host:display] cursor mask...\n",
#else
	"Usage: %s [-rv] [-fg color] [-bg color] [-window ID] [-select] [-interval N(msec)] [host:display] cursor mask...\n",
#endif
		cmd);
	exit(1);
}
