/*
** Copyright 1988 Silicon Graphics Inc.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation, and that the name of SGI not be used in advertising
** or publicity pertaining to distribution of the software without specific,
** written prior permission.  SGI makes no representations about the
** suitability of this software for any purpose.  It is provided "as is"
** without express or implied warranty.
**
** SGI DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL SGI
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
** OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
** CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
**
** Author:  Michael Toy, SGI
*/

/*
** These functions help the SGI IRIS's pretend they are old fashioned frame
** buffers by reading stuff off the screen in chunks as needed.
**
** We have two buffers which we use as a cache for blt operations.  Stuff is
** read off the screen into these buffers as needed and flushed back out
** when we are done.
**
** If the blt is from one window to itself, we will (Unfortunately) be reading
** the same bytes twice.
**
** The way this works is we read in rectangles of data from the screen
** which we hand copy into the scanline buffers, one line at time
** as the bitblt code goes through the rectangle.  Conversion from sgi
** format to X format is done as it is coped into the scanline buffer.
*/

#include "X.h"
#include "windowstr.h"
#include "pixmapstr.h"
#include "scrnintstr.h"
#include "cfb.h"
#include "cfbmskbits.h"
#include "servermd.h"
#include "sgifb.h"
#include "sgigl.h"

#define K		1024
#define CACHE_SIZE	(32*K)		/* # of pixels in each cache */

/*
** I hate people who #define C keywords, but I needed to do this for
** debugging purposes.
*/

#ifdef DEBUG
#define Private
#else
#define Private static
#endif

Private struct cached_drawable {
    DrawablePtr	cached;
    int		*pLine;			/* Pointer to line buffer */
#ifndef RECTREAD
    GLPixel	*pBuffer;		/* Pointer to rect buffer area */
    int		minY,			/* minY and maxY bound the cache */
		maxY;
#endif
    int		startY, endY,		/* Full size of blt */
		curY;			/* Current line in the cache */
    int		left;			/* Left edge */
    int		needed;			/* Need to read stuff */
    int		*base;			/* Base addr for un-cached pixmaps */
    int		pmwidth;		/* Width of lines for pixmaps */
} cache[2];

Private GLPixel	*pBuffer;		/* The buffer of GL Pixels */
Private int	fullHeight,		/* Total height of blt */
		bandHeight,		/* Part of fullHeight which fits */
		padWidth;		/* Padded width of one line */
int ForceBltToLoad;

/*
** Allocate the caches.  If XAlloc doesn't return long aligned memory,
** then we are in trouble.
*/
Private void
InitCache()
{
    if (!pBuffer) {
#ifndef RECTREAD
	pBuffer = (GLPixel *) Xalloc(2 * sizeof(GLPixel)*CACHE_SIZE);
#endif
	cache[BLT_SRC].pLine = (int *) Xalloc(SGI_WIDTH);
	cache[BLT_DST].pLine = (int *) Xalloc(SGI_WIDTH);
    }
}

/*
** Called at the start of a rectangular blt.  Returns a pointer to the
** line "initY" of the pixmap.  For windows, as much of the affected
** area as will fit in the cache is read in.  "num_drawables" tells whether
** we will be caching two or one drawables.
*/


void
sgiStartBlt(pSrc, pDst, w, h, alu)
    DrawablePtr pSrc, pDst;
{
    struct cached_drawable *src, *dst;
    char *srcBuf;

    InitCache();
    src = &cache[BLT_SRC];
    dst = &cache[BLT_DST];
#ifndef RECTREAD
    /*
    ** "padWidth" is big enough so that we can read a full word on either
    ** side of the blt regardless of alignment
    */
    padWidth = PixmapBytePad(w, PSZ) + sizeof (int);
    fullHeight = h;
    /*
    ** We try to be smart here.  If only one operand is a window, then
    ** we use the full cache for that.  Otherwise we divide it between
    ** the two.
    */
    bandHeight = 2 * CACHE_SIZE / padWidth;
    if (pDst->type == DRAWABLE_WINDOW) {
	dst->pBuffer = pBuffer;
	/*
	** This should say:
	** dst->needed = NEEDS_DST(alu);
	** but because I am reading extra at each end, I trash the screen
	** by doing that.  Should probably be fixed be only writing back
	** exactly the requested rectangle.  No time for that now.
	*/
	dst->needed = 1;
	dst->cached = pDst;
    } else {
	dst->base = (int *) (((PixmapPtr) pDst)->devPrivate);
	dst->pmwidth = (((PixmapPtr) pDst)->devKind) >> PWSH;
	dst->cached = 0;
    }
    if (pSrc)
	if (pSrc->type == DRAWABLE_WINDOW) {
	    if (dst->cached) {
		bandHeight /= 2;
		src->pBuffer = pBuffer + CACHE_SIZE;
	    } else
		src->pBuffer = pBuffer;
	    src->needed = NEEDS_SRC(alu);
	    src->cached = pSrc;
	} else {
	    src->base = (int *) (((PixmapPtr) pSrc)->devPrivate);
	    src->pmwidth = (((PixmapPtr) pSrc)->devKind) >> PWSH;
	    src->cached = 0;
	}
    else {
	src->base = 0;
	src->cached = 0;
    }
    if (bandHeight > h)
	bandHeight = h;
#else
    /*
    ** Clover1 graphics, no rectread -- ick.
    */
    padWidth = w;
    fullHeight = h;
    if (pDst->type == DRAWABLE_WINDOW) {
	dst->needed = 1;
	dst->cached = pDst;
    } else {
	dst->base = (int *) (((PixmapPtr) pDst)->devPrivate);
	dst->pmwidth = (((PixmapPtr) pDst)->devKind) >> PWSH;
	dst->cached = 0;
    }
    if (pSrc)
	if (pSrc->type == DRAWABLE_WINDOW) {
	    src->needed = NEEDS_SRC(alu);
	    src->cached = pSrc;
	} else {
	    src->base = (int *) (((PixmapPtr) pSrc)->devPrivate);
	    src->pmwidth = (((PixmapPtr) pSrc)->devKind) >> PWSH;
	    src->cached = 0;
	}
    else {
	src->base = 0;
	src->cached = 0;
    }
#endif
}

#ifndef RECTREAD
/*
** Read the cache'd band from the screen to the cache
*/

Private void
sgiCacheRectRead(cp)
    struct cached_drawable *cp;
{
    if (!cp->needed)	/* alu doesn't pay attention to source ... */
	return;		/* So don't bother reading it in */
    if (sgiBeginWindowDraw(cp->cached))
	return;
    /* Use readpixels for 1 pixel high rects on ECLIPSE ?? */
    GLrectread(cp->left, cp->minY, padWidth, bandHeight, (char*)cp->pBuffer);
}

/*
** Write the modified bad back out from the cache to the screen.
*/

Private void
sgiCacheRectWrite(cp)
    struct cached_drawable *cp;
{
    if (sgiBeginWindowDraw(cp->cached))
	return;
    GLrectwrite(cp->left, cp->minY, padWidth, bandHeight, (char*) cp->pBuffer);
}
#endif

/*
** Copy one line of the GL rectangle into an X format line.  Care is taken
** to put pixels at the same offset from cp->pLine as X would expect 
** to find them.
**
** This is addtionally complicated because the GL reads and write rectangles
** inverted in respect to the way that X expects them.
*/

Private void
sgiCopyOutLine(cp)
    struct cached_drawable *cp;
{
#ifdef RECTREAD
    GLgetpixels((char *) cp->pLine + cp->left, cp->left, cp->curY, padWidth);
#else
    register unsigned char *xpixel;
    register GLPixel *glpixel;
    register int i;

    if (!cp->needed)	/* alu doesn't pay attention to source ... */
	return;		/* So don't bother reading it in */
    xpixel = (unsigned char *)cp->pLine + cp->left;
    glpixel = cp->pBuffer + (bandHeight-(cp->curY-cp->minY)-1)*padWidth;
    for (i = 0; i < padWidth; i++)
	*xpixel++ = *glpixel++;
#endif
}

/*
** Copy the modified line back into the band cache.
*/

Private void
sgiCopyInLine(cp)
    struct cached_drawable *cp;
{
#ifdef RECTREAD
    GLputpixels((char *) cp->pLine + cp->left, cp->left, cp->curY, padWidth);
#else
    register unsigned char *xpixel;
    register GLPixel *glpixel;
    register int i;

    xpixel = (unsigned char *)cp->pLine + cp->left;
    glpixel = cp->pBuffer + (bandHeight-(cp->curY-cp->minY)-1)*padWidth;
    for (i = 0; i < padWidth; i++)
	*glpixel++ = *xpixel++ ;
#endif
}

/*
** This function is called to get the first address of a blt.
** for non cached blts, we get this from the pixmap.  Otherwise we
** set up for the read and return the cache line.
*/

int *
sgiBltAddr(which, x, y, initY)
{
    struct cached_drawable *cp;

    cp = &cache[which];
    if (!cp->cached) {
	return cp->base + cp->pmwidth*initY;
    }
#ifdef RECTREAD
    cp->left = x;
#else
    /*
    ** Pick up the full word containing the start pixel.
    */
    cp->left = x & ~PIM;
#endif
    /*
    ** Set up for correct direction of scanning
    */
    cp->startY = y;
    cp->endY = y + fullHeight - 1;
#ifndef RECTREAD
    if (y == initY) {
	cp->minY = y;
	cp->maxY = y + bandHeight - 1;
    } else {
	cp->minY = initY - bandHeight + 1;
	cp->maxY = initY;
    }
#endif
    cp->curY = initY;
    /*
    ** Load the cache and return.
    */
#ifndef RECTREAD
    sgiCacheRectRead(cp);
#endif
    sgiCopyOutLine(cp);
    return cp->pLine;
}

/*
** Called when finished with a line.  Replaces a call which said
**	pSrcLine += widthSrc
** For pixmaps we do just that, for windows, we return the passed in value
** of pSrcLine, except we've put new data in there.
**
** As a side effect we keep track of when to write out the bands.
**
** The points where 0 is returned are where we have finshed with the last line
** and cfb should never use those values.
*/

int *
sgiNextLine(which, pLine, width)
    int		which;
    int		*pLine;
    int		width;
{
    int delta;
    struct cached_drawable *cp;

    cp = &cache[which];
    if (!cp->cached)
	return pLine + width;
    sgiCopyInLine(cp);
#ifdef RECTREAD
    if ((cp->curY += delta) < cp->startY || cp->curY > cp->endY)
	return 0;
#else
    delta = (width > 0 ? 1 : -1);
    if ((cp->curY += delta) < cp->minY) {
	if (which == BLT_DST)
	    sgiCacheRectWrite(cp);
	if (cp->curY < cp->startY)
	    return 0;
	else {
	    int sy = cp->minY - bandHeight;

	    cp->maxY = cp->minY - 1;
	    if (sy < cp->startY)
		cp->minY = cp->startY;
	    else
		cp->minY = sy;
	    sgiCacheRectRead(cp);
	}
    } else if (cp->curY > cp->maxY) {
	if (which == BLT_DST)
	    sgiCacheRectWrite(cp);
	if (cp->curY > cp->endY)
	    return 0;
	else {
	    int ey = cp->maxY + bandHeight;

	    cp->minY = cp->maxY + 1;
	    if (ey > cp->endY)
		cp->maxY = cp->endY;
	    else
		cp->maxY = ey;
	    sgiCacheRectRead(cp);
	}
    }
#endif
    sgiCopyOutLine(cp);
    return pLine;
}

/*
** These routines are use by the span code rather than the rectangle
** oriented routines above.
*/

int *
sgiReadSpan(x, y, w, needed)
    int x, y, w;
{
    InitCache();
    if (needed) {
	int nx = x & ~PIM;

	w = PixmapBytePad(w + x - nx,PSZ);
	GLgetpixels((char *) cache[BLT_DST].pLine + nx, nx, y, w);
    }
    return cache[BLT_DST].pLine;
}

void
sgiWriteSpan(x, y, w)
    int x, y, w;
{
    GLputpixels((char *) cache[BLT_DST].pLine + x, x, y, w);
}
